"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// GLOBAL GIVEN
let stack;
let api;
let endpoint;
beforeEach(() => {
    stack = new cdk.Stack();
    api = new appsync.GraphQLApi(stack, 'baseApi', {
        name: 'api',
        schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
    });
    endpoint = 'aws.amazon.com';
});
describe('Http Data Source configuration', () => {
    test('default configuration produces name `HttpCDKDataSource`', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'ds',
        });
    });
    test('appsync configures name correctly', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint, {
            name: 'custom',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'custom',
        });
    });
    test('appsync configures name and description correctly', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint, {
            name: 'custom',
            description: 'custom description',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'custom',
            Description: 'custom description',
        });
    });
    test('appsync errors when creating multiple http data sources with no configuration', () => {
        // THEN
        expect(() => {
            api.addHttpDataSource('ds', endpoint);
            api.addHttpDataSource('ds', endpoint);
        }).toThrow("There is already a Construct with name 'ds' in GraphQLApi [baseApi]");
    });
});
describe('adding http data source from imported api', () => {
    test('imported api can add HttpDataSource from id', () => {
        // WHEN
        const importedApi = appsync.GraphQLApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
        });
        importedApi.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
    test('imported api can add HttpDataSource from attributes', () => {
        // WHEN
        const importedApi = appsync.GraphQLApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
            graphqlApiArn: api.arn,
        });
        importedApi.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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