"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const db = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
function joined(str) {
    return str.replace(/\s+/g, '');
}
// GLOBAL GIVEN
let stack;
let api;
beforeEach(() => {
    stack = new cdk.Stack();
    api = new appsync.GraphQLApi(stack, 'baseApi', {
        name: 'api',
        schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
    });
});
describe('DynamoDb Data Source configuration', () => {
    // GIVEN
    let table;
    beforeEach(() => {
        table = new db.Table(stack, 'table', {
            partitionKey: {
                name: 'id',
                type: db.AttributeType.STRING,
            },
        });
    });
    test('default configuration produces name `DynamoDbCDKDataSource`', () => {
        // WHEN
        api.addDynamoDbDataSource('ds', table);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AMAZON_DYNAMODB',
            Name: 'ds',
        });
    });
    test('appsync configures name correctly', () => {
        // WHEN
        api.addDynamoDbDataSource('ds', table, {
            name: 'custom',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AMAZON_DYNAMODB',
            Name: 'custom',
        });
    });
    test('appsync configures name and description correctly', () => {
        // WHEN
        api.addDynamoDbDataSource('ds', table, {
            name: 'custom',
            description: 'custom description',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AMAZON_DYNAMODB',
            Name: 'custom',
            Description: 'custom description',
        });
    });
    test('appsync errors when creating multiple dynamo db data sources with no configuration', () => {
        // THEN
        expect(() => {
            api.addDynamoDbDataSource('ds', table);
            api.addDynamoDbDataSource('ds', table);
        }).toThrow("There is already a Construct with name 'ds' in GraphQLApi [baseApi]");
    });
});
describe('DynamoDB Mapping Templates', () => {
    test('PutItem projecting all', () => {
        const template = appsync.MappingTemplate.dynamoDbPutItem(appsync.PrimaryKey.partition('id').is('id'), appsync.Values.projecting());
        const rendered = joined(template.renderTemplate());
        expect(rendered).toStrictEqual(joined(`
      #set($input = $ctx.args)
      {
        "version" : "2017-02-28",
        "operation" : "PutItem",
        "key" : {
          "id" : $util.dynamodb.toDynamoDBJson($ctx.args.id)
        },
        "attributeValues": $util.dynamodb.toMapValuesJson($input)
      }`));
    });
    test('PutItem with invididual attributes', () => {
        const template = appsync.MappingTemplate.dynamoDbPutItem(appsync.PrimaryKey.partition('id').is('id'), appsync.Values.attribute('val').is('ctx.args.val'));
        const rendered = joined(template.renderTemplate());
        expect(rendered).toStrictEqual(joined(`
      #set($input = {})
      $util.qr($input.put("val", ctx.args.val))
      {
        "version" : "2017-02-28",
        "operation" : "PutItem",
        "key" : {
          "id" : $util.dynamodb.toDynamoDBJson($ctx.args.id)
        },
        "attributeValues": $util.dynamodb.toMapValuesJson($input)
      }`));
    });
    test('PutItem with additional attributes', () => {
        const template = appsync.MappingTemplate.dynamoDbPutItem(appsync.PrimaryKey.partition('id').is('id'), appsync.Values.projecting().attribute('val').is('ctx.args.val'));
        const rendered = joined(template.renderTemplate());
        expect(rendered).toStrictEqual(joined(`
      #set($input = $ctx.args)
      $util.qr($input.put("val", ctx.args.val))
      {
        "version" : "2017-02-28",
        "operation" : "PutItem",
        "key" : {
          "id" : $util.dynamodb.toDynamoDBJson($ctx.args.id)
        },
        "attributeValues": $util.dynamodb.toMapValuesJson($input)
      }`));
    });
});
describe('adding DynamoDb data source from imported api', () => {
    // GIVEN
    let table;
    beforeEach(() => {
        table = new db.Table(stack, 'table', {
            partitionKey: {
                name: 'id',
                type: db.AttributeType.STRING,
            },
        });
    });
    test('imported api can add DynamoDbDataSource from id', () => {
        // WHEN
        const importedApi = appsync.GraphQLApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
        });
        importedApi.addDynamoDbDataSource('ds', table);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AMAZON_DYNAMODB',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
    test('imported api can add DynamoDbDataSource from attributes', () => {
        // WHEN
        const importedApi = appsync.GraphQLApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
            graphqlApiArn: api.arn,
        });
        importedApi.addDynamoDbDataSource('ds', table);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AMAZON_DYNAMODB',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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