"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: core_1.Lazy.stringValue({ produce: () => this.definition }),
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.addToSchema(core_1.Lazy.stringValue({ produce: () => type.toString() }));
        return type;
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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