"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectType = exports.InterfaceType = void 0;
const schema_field_1 = require("./schema-field");
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let schemaAddition = `interface ${this.name} {\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            const args = attribute.argsToString();
            schemaAddition = `${schemaAddition}  ${key}${args}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the field to add
     */
    addField(fieldName, field) {
        this.definition[fieldName] = field;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.directives = props.directives;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the resolvable field to add
     */
    addField(fieldName, field) {
        this.generateResolver(fieldName, field.fieldOptions);
        this.definition[fieldName] = field;
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let title = this.name;
        if (this.interfaceTypes && this.interfaceTypes.length) {
            title = `${title} implements`;
            this.interfaceTypes.map((interfaceType) => {
                title = `${title} ${interfaceType.name},`;
            });
            title = title.slice(0, -1);
        }
        const directives = this.generateDirectives(this.directives);
        let schemaAddition = `type ${title} ${directives}{\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            const args = attribute.argsToString();
            schemaAddition = `${schemaAddition}  ${key}${args}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
    /**
     * Utility function to generate directives
     *
     * @param directives the directives of a given type
     * @param delimiter the separator betweeen directives
     * @default - ' '
     */
    generateDirectives(directives, delimiter) {
        let schemaAddition = '';
        if (!directives) {
            return schemaAddition;
        }
        directives.map((directive) => {
            schemaAddition = `${schemaAddition}${directive.statement}${delimiter !== null && delimiter !== void 0 ? delimiter : ' '}`;
        });
        return schemaAddition;
    }
    /**
     * Generate the resolvers linked to this Object Type
     */
    generateResolver(fieldName, options) {
        if (options === null || options === void 0 ? void 0 : options.dataSource) {
            if (!this.resolvers) {
                this.resolvers = [];
            }
            this.resolvers.push(options.dataSource.createResolver({
                typeName: this.name,
                fieldName: fieldName,
                pipelineConfig: options.pipelineConfig,
                requestMappingTemplate: options.requestMappingTemplate,
                responseMappingTemplate: options.responseMappingTemplate,
            }));
        }
    }
}
exports.ObjectType = ObjectType;
//# sourceMappingURL=data:application/json;base64,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