import { Resolver } from './resolver';
import { Directive, IField, IIntermediateType } from './schema-base';
import { BaseTypeOptions, GraphqlType, ResolvableFieldOptions } from './schema-field';
/**
 * Properties for configuring an Intermediate Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 *
 * @experimental
 */
export interface IntermediateTypeProps {
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
}
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
export declare class InterfaceType implements IIntermediateType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    constructor(name: string, props: IntermediateTypeProps);
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options?: BaseTypeOptions): GraphqlType;
    /**
     * Generate the string of this object type
     */
    toString(): string;
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the field to add
     */
    addField(fieldName: string, field: IField): void;
}
/**
 * Properties for configuring an Object Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 * @param interfaceTypes - the interfaces that this object type implements
 * @param directives - the directives for this object type
 *
 * @experimental
 */
export interface ObjectTypeProps extends IntermediateTypeProps {
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
}
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
export declare class ObjectType extends InterfaceType implements IIntermediateType {
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
    /**
     * The resolvers linked to this data source
     */
    resolvers?: Resolver[];
    constructor(name: string, props: ObjectTypeProps);
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the resolvable field to add
     */
    addField(fieldName: string, field: IField): void;
    /**
     * Generate the string of this object type
     */
    toString(): string;
    /**
     * Utility function to generate directives
     *
     * @param directives the directives of a given type
     * @param delimiter the separator betweeen directives
     * @default - ' '
     */
    private generateDirectives;
    /**
     * Generate the resolvers linked to this Object Type
     */
    protected generateResolver(fieldName: string, options?: ResolvableFieldOptions): void;
}
