"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolvableField = exports.Field = exports.GraphqlType = void 0;
const schema_base_1 = require("./schema-base");
/**
 * The GraphQL Types in AppSync's GraphQL. GraphQL Types are the
 * building blocks for object types, queries, mutations, etc. They are
 * types like String, Int, Id or even Object Types you create.
 *
 * i.e. `String`, `String!`, `[String]`, `[String!]`, `[String]!`
 *
 * GraphQL Types are used to define the entirety of schema.
 */
class GraphqlType {
    constructor(type, options) {
        var _a, _b, _c;
        this.type = type;
        this.isList = (_a = options === null || options === void 0 ? void 0 : options.isList) !== null && _a !== void 0 ? _a : false;
        this.isRequired = (_b = options === null || options === void 0 ? void 0 : options.isRequired) !== null && _b !== void 0 ? _b : false;
        this.isRequiredList = (_c = options === null || options === void 0 ? void 0 : options.isRequiredList) !== null && _c !== void 0 ? _c : false;
        this.intermediateType = options === null || options === void 0 ? void 0 : options.intermediateType;
    }
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static id(options) {
        return new GraphqlType(schema_base_1.Type.ID, options);
    }
    /**
     * `String` scalar type is a free-form human-readable text.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static string(options) {
        return new GraphqlType(schema_base_1.Type.STRING, options);
    }
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static int(options) {
        return new GraphqlType(schema_base_1.Type.INT, options);
    }
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static float(options) {
        return new GraphqlType(schema_base_1.Type.FLOAT, options);
    }
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static boolean(options) {
        return new GraphqlType(schema_base_1.Type.BOOLEAN, options);
    }
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDate(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE, options);
    }
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIME, options);
    }
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDateTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE_TIME, options);
    }
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTimestamp(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIMESTAMP, options);
    }
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsEmail(options) {
        return new GraphqlType(schema_base_1.Type.AWS_EMAIL, options);
    }
    /**
     * `AWSJson` scalar type represents a JSON string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsJson(options) {
        return new GraphqlType(schema_base_1.Type.AWS_JSON, options);
    }
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsUrl(options) {
        return new GraphqlType(schema_base_1.Type.AWS_URL, options);
    }
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsPhone(options) {
        return new GraphqlType(schema_base_1.Type.AWS_PHONE, options);
    }
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsIpAddress(options) {
        return new GraphqlType(schema_base_1.Type.AWS_IP_ADDRESS, options);
    }
    /**
     * an intermediate type to be added as an attribute
     * (i.e. an interface or an object type)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     * - intermediateType
     */
    static intermediate(options) {
        if (!(options === null || options === void 0 ? void 0 : options.intermediateType)) {
            throw new Error('GraphQL Type of interface must be configured with corresponding Intermediate Type');
        }
        return new GraphqlType(schema_base_1.Type.INTERMEDIATE, options);
    }
    /**
     * Generate the string for this attribute
     */
    toString() {
        var _a;
        // If an Object Type, we use the name of the Object Type
        let type = this.intermediateType ? (_a = this.intermediateType) === null || _a === void 0 ? void 0 : _a.name : this.type;
        // If configured as required, the GraphQL Type becomes required
        type = this.isRequired ? `${type}!` : type;
        // If configured with isXxxList, the GraphQL Type becomes a list
        type = this.isList || this.isRequiredList ? `[${type}]` : type;
        // If configured with isRequiredList, the list becomes required
        type = this.isRequiredList ? `${type}!` : type;
        return type;
    }
    /**
     * Generate the arguments for this field
     */
    argsToString() {
        return '';
    }
}
exports.GraphqlType = GraphqlType;
/**
 * Fields build upon Graphql Types and provide typing
 * and arguments.
 */
class Field extends GraphqlType {
    constructor(options) {
        const props = {
            isList: options.returnType.isList,
            isRequired: options.returnType.isRequired,
            isRequiredList: options.returnType.isRequiredList,
            intermediateType: options.returnType.intermediateType,
        };
        super(options.returnType.type, props);
        this.fieldOptions = options;
    }
    /**
     * Generate the args string of this resolvable field
     */
    argsToString() {
        var _a, _b;
        if (!this.fieldOptions || !this.fieldOptions.args) {
            return '';
        }
        let args = '(';
        Object.keys((_b = (_a = this.fieldOptions) === null || _a === void 0 ? void 0 : _a.args) !== null && _b !== void 0 ? _b : {}).forEach((key) => {
            var _a, _b;
            const type = (_b = (_a = this.fieldOptions) === null || _a === void 0 ? void 0 : _a.args) === null || _b === void 0 ? void 0 : _b[key].toString();
            args = `${args}${key}: ${type} `;
        });
        args = args.slice(0, -1);
        return `${args})`;
    }
}
exports.Field = Field;
/**
 * Resolvable Fields build upon Graphql Types and provide fields
 * that can resolve into operations on a data source.
 */
class ResolvableField extends Field {
    constructor(options) {
        const props = {
            returnType: options.returnType,
            args: options.args,
        };
        super(props);
        this.fieldOptions = options;
    }
}
exports.ResolvableField = ResolvableField;
//# sourceMappingURL=data:application/json;base64,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