"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.SchemaMode = void 0;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility enum for Schema class
 */
var SchemaMode;
(function (SchemaMode) {
    SchemaMode["FILE"] = "FILE";
    SchemaMode["CODE"] = "CODE";
})(SchemaMode = exports.SchemaMode || (exports.SchemaMode = {}));
;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
//# sourceMappingURL=data:application/json;base64,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