"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphQLApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultMode = (_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) !== null && _b !== void 0 ? _b : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.additionalAuthorizationModes) !== null && _d !== void 0 ? _d : [];
        const modes = [defaultMode, ...additionalModes];
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_e = props.schema) !== null && _e !== void 0 ? _e : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_f = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _f === void 0 ? void 0 : _f.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const role = new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        });
        return {
            cloudWatchLogsRoleArn: role.roleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        let expires;
        if (config === null || config === void 0 ? void 0 : config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        return this.schema.addType(type);
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,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