"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a, _b;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        const name = (_a = props.name) !== null && _a !== void 0 ? _a : id;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: name,
            description: props.description,
            serviceRoleArn: (_b = this.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            ...extended,
        });
        this.name = name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
//# sourceMappingURL=data:application/json;base64,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