# -*- coding: utf-8 -*-
from .api import ListoAPI
from time import sleep


class Documents(ListoAPI):
    def __init__(self, token, base_url):
        super(Documents, self).__init__(token, base_url)

    def get_document(self, document_id, doc_name):
        """Download original XML from an invoice or pdf

        Args:
            - document_id (int|str): Id of document. Found at Invoice.documents.files[i].id
            - doc_name (str): Name and path to save the document

        More information at https://apidocs.listo.mx/#descarga-de-xml
        """
        with open(doc_name, 'wb') as f:
            f.write(self.make_request(method="GET", path="/documents/full/%s" % document_id).content)

    def zip_bulk_xml(self, zip_name, **kwargs):
        """Download all of the XMLs for some invoices

        KwArgs:
        - invoices.search
        """
        # Validate date filters
        for d in ["issued_on", "due_on", "modified_on", "canceled_on"]:
            if kwargs.get(d):
                self.validate_date_filter(kwargs.get(d))

        key = self.make_request(method="POST", path="/invoices/export_zip",
                                data=kwargs).json()['key']
        status = "pending"
        while status == "pending":
            r = self.make_request(method="GET", path="/invoices/export_status/%s" % key).json()
            status = r['status']

        with open(zip_name, 'wb') as f:
            f.write(self.make_request(method="GET", url=r['url'], set_header=False).content)

    def zip_bulk_pdf(self, zip_name, **kwargs):
        """Download all of the existing PDFs for some invoices

        KwArgs:
        - invoices.search
        """
        # Validate date filters
        for d in ["issued_on", "due_on", "modified_on", "canceled_on"]:
            if kwargs.get(d):
                self.validate_date_filter(kwargs.get(d))

        key = self.make_request(method="POST", path="/invoices/export_pdf_zip",
                                data=kwargs).json()['key']
        status = "pending"
        while status == "pending":
            r = self.make_request(method="GET", path="/invoices/export_status/%s" % key).json()
            status = r['status']
            sleep(1)

        with open(zip_name, 'wb') as f:
            f.write(self.make_request(method="GET", url=r['url'], set_header=False).content)

    def get_pdf(self, document_id, pdf_name, document_type='invoices'):
        """Download pdf from invoices or receipts

        Args:
            - document_id (str|int): Invoice_id or Receipt_id
            - pdf_name: path where the file will be saved including name and extension
            - document_type (str):  \'invoices\' or \'receipts\'
        """
        document_type = document_type.lower()
        if document_type not in ['invoices', 'receipts']:
            raise Exception('document_type not allowed')

        if document_type == 'receipts':
            path = "/cfdi_payments/receipts/%s/autogenerated_pdf" % document_id
        else:
            path = "/invoices/%s/autogenerated_pdf" % document_id

        with open(pdf_name, 'wb') as f:
            f.write(self.make_request(method="GET", path=path).content)
