import numpy as np
from scipy.optimize import root
from ..math import gdem
from .equilibriumresult import EquilibriumResult

#ELV phi-phi
def bubble_sus(P_T,X,T_P,tipo,y_guess,eos, vl0, vv0):
    
    if tipo == 'T':
        P=P_T
        T=T_P
    elif tipo == 'P':
        T=P_T
        P=T_P
    
    #fugacidades de liquido
    lnphil, vl =eos.logfugef(X,T,P,'L', vl0)
    
    tol=1e-8
    error=1
    itacc = 0
    niter=0
    n = 5
    Y_calc=y_guess
    Y=y_guess

    while error > tol and itacc < 3:
        niter+=1
        
        #calculo de fugacidad de la fase vapor
        lnphiv, vv0 = eos.logfugef(Y,T,P,'V', vv0)
        
        lnK = lnphil-lnphiv
        K=np.exp(lnK)
        Y_calc_old=Y_calc
        Y_calc=X*K

        if niter == (n-3):
            Y3 = Y_calc
        elif niter == (n-2):
            Y2 = Y_calc
        elif niter == (n-1):
            Y1 = Y_calc
        elif niter == n:
            niter = 0 
            itacc += 1
            dacc = gdem(Y, Y1, Y2, Y3)
            Y_calc += dacc
        error=((Y_calc-Y_calc_old)**2).sum()
        Y=Y_calc/Y_calc.sum()
    
    if tipo == 'T':
        f0 = Y_calc.sum() - 1
    elif tipo == 'P':
        f0 = np.log(Y_calc.sum())
       
    return f0, Y, lnK, vl, vv0

def bubble_newton(inc,X,T_P,tipo, eos, vl0, vv0):
    global vl, vv
    f = np.zeros_like(inc)
    lnK = inc[:-1]
    K = np.exp(lnK) 
    
    if tipo == 'T':
        P = inc[-1]
        T = T_P
    elif tipo == 'P':
        T = inc[-1]
        P = T_P
    
    Y = X*K
    
    #fugacidades de liquido
    lnphil, vl = eos.logfugef(X,T,P,'L', vl0)
    #calculo de fugacidad de la fase vapor
    lnphiv, vv = eos.logfugef(Y,T,P,'V', vv0)
    
    f[:-1] = lnK + lnphiv - lnphil
    f[-1] = (Y-X).sum()
    
    return f


def bubblePy(y_guess, P_guess, X, T, model, good_initial = False,
             v0 = [None, None] , full_output = False):
    
    """
    Bubble point (T, x) -> (P, y)
    
    Inputs
    ----------
    y_guess : array_like, guess of vapour phase composition
    P_guess : guess of equilibrium pressure in bar.
    
    x : array_like, liquid phase composition
    T : absolute temperature of the liquid in K.
    model : object create from mixture, eos and mixrule 
    
    """
    global vl, vv
    vl0, vv0 = v0
    
    it = 0
    itmax = 10
    tol = 1e-8
    
    P = P_guess
    f, Y, lnK, vl, vv = bubble_sus(P, X, T,'T',y_guess, model, vl0, vv0)
    error = np.abs(f)
    h = 1e-4
    
    while error > tol and it <= itmax and not good_initial:
        it += 1
        f, Y, lnK, vl, vv = bubble_sus(P, X, T, 'T', Y, model, vl, vv)
        f1, Y1, lnK1, vl, vv = bubble_sus( P + h ,X, T,'T',Y, model, vl, vv)
        df = (f1-f)/h
        P -= f/df
        error = np.abs(f)
    
    if error > tol:       
        inc0 = np.hstack([lnK, P])
        sol1 = root(bubble_newton, inc0, args = (X, T,'T',model, vl, vv))
        sol = sol1.x
        lnK = sol[:-1]
        error = np.linalg.norm(sol1.fun)
        it += sol1.nfev
        Y = np.exp(lnK)*X
        Y /= Y.sum()
        P = sol[-1]
        
    if full_output:
        sol = {'T' : T, 'P': P, 'error':error, 'iter':it,
        'X' : X, 'v1':vl, 'state1' : 'Liquid',
       'Y' : Y, 'v2': vv, 'state2' : 'Vapor'}
        out = EquilibriumResult(sol)
        return out    
        
        
    return Y, P


def bubbleTy(y_guess, T_guess, X, P, model, good_initial = False,
             v0 = [None, None], full_output = False):
    """
    Bubble point (P, x) -> (T, y)
    
    Inputs
    ----------
    y_guess : array_like, guess of vapour phase composition
    T_guess : guess of equilibrium temperature of the liquid in K.
    
    x : array_like, liquid phase composition
    P : pressure of the liquid in bar
    model : object create from mixture, eos and mixrule 
    
    """
    global vl, vv
    
    vl0, vv0 = v0
        
    it = 0
    itmax = 10
    tol = 1e-8
    
    T = T_guess
    f, Y, lnK, vl, vv = bubble_sus(T,X,P,'P',y_guess,model,vl0, vv0)
    error = np.abs(f)
    h = 1e-4
    
    while error > tol and it <= itmax and not good_initial:
        it += 1
        f1, Y1, lnK1, vl, vv = bubble_sus( T + h ,X,P,'P',Y,model, vl, vv)
        f, Y, lnK, vl, vv = bubble_sus(T,X,P,'P',Y,model, vl, vv)
        df = (f1-f)/(h)
        T -= f/df
        error = np.abs(f)
    
    if error > tol:       
        inc0 = np.hstack([lnK, T])
        sol1 = root(bubble_newton, inc0, args = (X,P,'P',model, vl, vv))
        sol = sol1.x
        lnK = sol[:-1]
        error = np.linalg.norm(sol1.fun)
        it += sol1.nfev
        Y = np.exp(lnK)*X
        Y /= Y.sum()
        T = sol[-1]
    
    if full_output:
        sol = {'T' : T, 'P': P, 'error':error, 'iter':it,
        'X' : X, 'v1':vl, 'state1' : 'Liquid',
       'Y' : Y, 'v2': vv, 'state2' : 'Vapor'}
        out = EquilibriumResult(sol)
        return out    
        
    return Y, T

__all__ = ['bubbleTy', 'bubblePy']