from __future__ import annotations

import argparse
import dataclasses
import enum
import functools
import itertools
import shlex
from typing import Any, Dict, Mapping, Optional, Set, Tuple, Type, TypeVar, Union

from . import _fields, _instantiators

try:
    # Python >=3.8.
    from functools import cached_property
except ImportError:
    # Python 3.7.
    from backports.cached_property import cached_property  # type: ignore


@dataclasses.dataclass(frozen=True)
class ArgumentDefinition:
    """Structure containing everything needed to define an argument."""

    prefix: str  # Prefix for nesting.
    field: _fields.Field
    type_from_typevar: Dict[TypeVar, Type]

    def add_argument(
        self, parser: Union[argparse.ArgumentParser, argparse._ArgumentGroup]
    ) -> None:
        """Add a defined argument to a parser."""

        # Get keyword arguments, with None values removed.
        kwargs = dataclasses.asdict(self.lowered)
        kwargs = dict(filter(lambda kv: kv[1] is not None, kwargs.items()))

        # Pop lowered properties t
        kwargs.pop("instantiator")
        name_or_flag = kwargs.pop("name_or_flag")

        # Note that the name must be passed in as a position argument.
        parser.add_argument(name_or_flag, **kwargs)

    @cached_property
    def lowered(self) -> LoweredArgumentDefinition:
        """Lowered argument definition, generated by applying a sequence of rules."""
        rules = (
            _rule_handle_defaults,
            _rule_handle_boolean_flags,
            _rule_recursive_instantiator_from_type,
            _rule_convert_defaults_to_strings,
            _rule_generate_helptext,
            _rule_set_name_or_flag,
            _rule_positional_special_handling,
        )
        return functools.reduce(
            lambda lowered, rule: rule(self, lowered),
            rules,
            LoweredArgumentDefinition(),
        )


@dataclasses.dataclass(frozen=True)
class LoweredArgumentDefinition:
    """Contains fields meant to be passed directly into argparse."""

    # Action that is called on parsed arguments. This handles conversions from strings
    # to our desired types.
    #
    # The main reason we use this instead of the standard 'type' argument is to enable
    # mixed-type tuples.
    instantiator: Optional[_instantiators.Instantiator] = None

    # From here on out, all fields correspond 1:1 to inputs to argparse's
    # add_argument() method.
    name_or_flag: str = ""
    default: Optional[Any] = None
    dest: Optional[str] = None
    required: bool = False
    action: Optional[str] = None
    nargs: Optional[Union[int, str]] = None
    choices: Optional[Set[Any]] = None
    metavar: Optional[Union[str, Tuple[str, ...]]] = None
    help: Optional[str] = None


def _rule_handle_defaults(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    """Set `required=True` if a default value is set."""

    # Mark lowered as required if a default is set.
    if arg.field.default is None:
        return dataclasses.replace(lowered, required=True)

    return dataclasses.replace(lowered, default=arg.field.default)


def _rule_handle_boolean_flags(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    if arg.type_from_typevar.get(arg.field.typ, arg.field.typ) is not bool:  # type: ignore
        return lowered

    if lowered.default is False and not arg.field.positional:
        # Default `False` => --flag passed in flips to `True`.
        return dataclasses.replace(
            lowered,
            action="store_true",
            instantiator=lambda x: x,  # argparse will directly give us a bool!
        )
    elif lowered.default is True and not arg.field.positional:
        # Default `True` => --no-flag passed in flips to `False`.
        return dataclasses.replace(
            lowered,
            action="store_false",
            instantiator=lambda x: x,  # argparse will directly give us a bool!
        )
    else:
        # Treat bools as a normal parameter.
        return lowered


def _rule_recursive_instantiator_from_type(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    """The bulkiest bit: recursively analyze the type annotation and use it to determine
    how to instantiate it given some string from the commandline.

    Important: as far as argparse is concerned, all inputs are strings.

    Conversions from strings to our desired types happen in the instantiator; this is a
    bit more flexible, and lets us handle more complex types like enums and multi-type
    tuples."""
    if lowered.instantiator is not None:
        return lowered

    instantiator, metadata = _instantiators.instantiator_from_type(
        arg.field.typ,  # type: ignore
        arg.type_from_typevar,
    )
    return dataclasses.replace(
        lowered,
        instantiator=instantiator,
        choices=metadata.choices,
        nargs=metadata.nargs,
        required=(not metadata.is_optional) and lowered.required,
        metavar=metadata.metavar,
    )


def _rule_convert_defaults_to_strings(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    """Sets all default values to strings, as required as input to our instantiator
    functions. Special-cased for enums."""

    def as_str(x: Any) -> str:
        if isinstance(x, enum.Enum):
            return x.name
        else:
            return str(x)

    if lowered.default is None or lowered.action is not None:
        return lowered
    elif lowered.nargs is not None and lowered.nargs != "?":
        if isinstance(lowered.default, Mapping):
            return dataclasses.replace(
                lowered,
                default=tuple(map(as_str, itertools.chain(*lowered.default.items()))),
            )
        else:
            return dataclasses.replace(
                lowered, default=tuple(map(as_str, lowered.default))
            )
    else:
        return dataclasses.replace(lowered, default=as_str(lowered.default))


def _rule_generate_helptext(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    """Generate helptext from docstring, argument name, default values."""
    help_parts = []

    docstring_help = arg.field.helptext

    if docstring_help is not None and docstring_help != "":
        # Note that the percent symbol needs some extra handling in argparse.
        # https://stackoverflow.com/questions/21168120/python-argparse-errors-with-in-help-string
        docstring_help = docstring_help.replace("%", "%%")
        help_parts.append(docstring_help)
    elif arg.field.positional:
        help_parts.append(str(lowered.metavar))

    if lowered.action is not None:
        # Don't show defaults for boolean flags.
        # assert lowered.action in ("store_true", "store_false")
        pass
    elif not lowered.required:
        # Include default value in helptext. We intentionally don't use the % template
        # because the types of all arguments are set to strings, which will cause the
        # default to be casted to a string and introduce extra quotation marks.
        if lowered.nargs is not None and hasattr(lowered.default, "__iter__"):
            # For tuple types, we might have lowered.default as (0, 1, 2, 3).
            # For list types, we might have lowered.default as [0, 1, 2, 3].
            # For set types, we might have lowered.default as {0, 1, 2, 3}.
            #
            # In all cases, we want to display (default: 0 1 2 3), for consistency with
            # the format that argparse expects when we set nargs.
            assert lowered.default is not None  # Just for type checker.
            default_parts = map(shlex.quote, map(str, lowered.default))
            help_parts.append(f"(default: {' '.join(default_parts)})")
        else:
            help_parts.append(f"(default: {shlex.quote(str(lowered.default))})")

    return dataclasses.replace(lowered, help=" ".join(help_parts))


def _rule_set_name_or_flag(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    if arg.field.positional:
        name_or_flag = arg.prefix + arg.field.name
    elif lowered.action == "store_false":
        name_or_flag = "--" + (arg.prefix + "no-" + arg.field.name).replace("_", "-")
    else:
        name_or_flag = "--" + (arg.prefix + arg.field.name).replace("_", "-")

    return dataclasses.replace(
        lowered, name_or_flag=name_or_flag, dest=arg.prefix + arg.field.name
    )


def _rule_positional_special_handling(
    arg: ArgumentDefinition,
    lowered: LoweredArgumentDefinition,
) -> LoweredArgumentDefinition:
    if not arg.field.positional:
        return lowered

    return dataclasses.replace(
        lowered,
        dest=None,
        metavar=(arg.prefix + arg.field.name).upper(),
        required=None,
        nargs="?" if not lowered.required else lowered.nargs,
    )
