"""Mloq license command implementation."""
from pathlib import Path

import click
from omegaconf import DictConfig

from mloq.command import Command
from mloq.config.param_patch import param
from mloq.files import ASSETS_PATH, file


TEMPLATES_PATH = ASSETS_PATH / "license"
dco = file(
    "DCO.md",
    TEMPLATES_PATH,
    "Developer Certificate of Origin - needed in open source projects to certify that "
    "the incoming contributions are legitimate",
    is_static=True,
)
mit_license = file("MIT_LICENSE", TEMPLATES_PATH, "license of the project", dst="LICENSE")
apache_license = file("APACHE_LICENSE", TEMPLATES_PATH, "license of the project", dst="LICENSE")
gpl_license = file(
    "GPL_LICENSE",
    TEMPLATES_PATH,
    "license of the project",
    dst="LICENSE",
    is_static=True,
)
LICENSES = {
    "MIT": mit_license,
    "Apache-2.0": apache_license,
    "GPL-3.0": gpl_license,
}

LICENSE_FILES = [file for file in LICENSES.values()] + [dco]


class LicenseCMD(Command):
    """Implement the functionality of the license Command."""

    cmd_name = "license"
    files = tuple(LICENSE_FILES)
    LICENSES = LICENSES
    disable = param.Boolean(default=None, doc="Disable license command?")
    license = param.String("MIT", doc="Project license type")
    copyright_year = param.Integer("${current_year:}", doc="Year when the project started")
    copyright_holder = param.String("${globals.owner}", doc="Copyright holder")
    project_name = param.String(doc="Select project name")
    project_url = param.String("${globals.project_url}", doc="GitHub project url")
    email = param.String("${globals.email}", doc="Owner contact email")

    def interactive_config(self) -> DictConfig:
        """Generate the configuration of the project interactively."""
        click.echo("Provide the values to generate the project license files.")
        return self.parse_config()

    def record_files(self) -> None:
        """Register the files that will be generated by mloq."""
        conf = self.record.config.license
        self.record.register_file(file=dco, path=Path())
        if conf.license != "proprietary":
            license_file = self.LICENSES[conf.license]
            self.record.register_file(file=license_file, path=Path())
