"""Mloq docs command implementation."""
from pathlib import Path
from typing import Tuple

import click
from omegaconf import DictConfig
import param

from mloq.command import Command
from mloq.files import ASSETS_PATH, file


DOCS_ASSETS_PATH = ASSETS_PATH / "docs"
# Documentation files
makefile_docs = file(
    "makefile_docs.txt",
    DOCS_ASSETS_PATH,
    "common make commands for building the documentation",
    dst="Makefile",
    is_static=True,
)
make_bat_docs = file(
    "make_bat.txt",
    DOCS_ASSETS_PATH,
    "common make commands for building the documentation",
    dst="make.bat",
    is_static=True,
)
docs_req = file(
    "requirements-docs.txt",
    DOCS_ASSETS_PATH,
    "list of exact versions of the packages needed to build your documentation",
    is_static=True,
)
conf_py = file(
    "conf.txt",
    DOCS_ASSETS_PATH,
    "configuration file for sphinx and doc plugins",
    dst="conf.py",
)
index_md = file(
    "index.md",
    DOCS_ASSETS_PATH,
    "configuration file for sphinx and doc plugins",
)
deploy_docs = file(
    "deploy-docs.yml",
    DOCS_ASSETS_PATH,
    "workflow for deploying the documentation to GitHub pages",
)

DOCS_FILES = [conf_py, index_md, makefile_docs, make_bat_docs, docs_req, deploy_docs]


class DocsCMD(Command):
    """Implement the functionality of the docs Command."""

    cmd_name = "docs"
    disable = param.Boolean(False, doc="Disable docs command?")
    project_name = param.String(doc="Select project name")
    description = param.String(doc="Short description of the project")
    author = param.String("${globals.author}", doc="Author(s) of the project")
    copyright_year = param.Integer(doc="Year when the project started")
    copyright_holder = param.String("${docs.author}", doc="Copyright holder")
    deploy_docs = param.Boolean(True, doc="Deploy docs to GitHub Pages?")
    default_branch = param.String("${globals.default_branch}", doc="Branch used to build the docs")
    project_url = param.String("${globals.project_url}", doc="GitHub project url")
    files = tuple(DOCS_FILES)

    @property
    def directories(self) -> Tuple[Path]:
        """Tuple containing Paths objects representing the directories created by the command."""
        docs_folder = Path("docs") / "source" / "markdown"
        paths = [docs_folder]
        if self.deploy_docs:
            workflow_path = Path(".github") / "workflows"
            paths.append(workflow_path)
        return tuple(paths)

    def interactive_config(self) -> DictConfig:
        """Generate the configuration of the project interactively."""
        click.echo("Provide the values to generate the project documentation.")
        return super(DocsCMD, self).interactive_config()

    def record_files(self) -> None:
        """Register the files that will be generated by mloq."""
        source_files = {"conf.py", "index.md"}
        docs_ASSETS_path = Path("docs")
        for _file in self.files:
            if _file == deploy_docs:
                if not self.deploy_docs:
                    continue
                path = Path(".github") / "workflows"
            else:
                path = (
                    (docs_ASSETS_path / "source")
                    if str(_file.dst) in source_files
                    else docs_ASSETS_path
                )
            self.record.register_file(file=_file, path=path)
