from collections.abc import Iterable
from typing import Final, Literal, Optional, Union
from urllib.parse import urlsplit

import httpx
from pydantic import BaseModel

from . import schema

Action = Literal["adjust", "allocate", "deallocate", "receive", "release"]
post_headers: Final = {"Content-Type": "application/json", "Accept": "application/json"}
SESSION_HEADER_KEY: Final = "X-Inventory-Session"


# TODO: Ensure this model can be autogenerated in the schema.py file
class StockList(BaseModel):
    __root__: Union[list[schema.Stock], list[schema.StockAdjustment]]


class Inventory:
    def __init__(self, client: httpx.Client) -> None:
        self.client = client

    def list_available_stock(
        self, warehouse_code: str, skus: Iterable[str], limit: int = 100
    ) -> Iterable[schema.StockAvailability]:
        skus_query = str(httpx.QueryParams({"sku": list(skus)}).add("limit", limit))
        next_page = f"/private/api/v1/inventory/{warehouse_code}/?{skus_query}"
        while True:
            response = self.client.get(
                next_page, headers={"Accept": "application/json"}
            )
            response.raise_for_status()
            data = schema.PaginatedStockAvailabilityList.parse_obj(response.json())
            yield from data.results or []

            next_page = (
                "{0.path}?{0.query}".format(urlsplit(data.next)) if data.next else ""
            )
            if not next_page:
                break

    def _post_quantity(
        self,
        action: Action,
        /,
        warehouse_code: str,
        data: Union[Iterable[schema.Stock], Iterable[schema.StockAdjustment]],
        session: Optional[str] = None,
    ) -> None:
        headers = dict(post_headers)
        if session is not None:
            headers[SESSION_HEADER_KEY] = session
        self.client.post(
            f"/private/api/v1/inventory/{warehouse_code}/{action}/",
            content=StockList.parse_obj({"__root__": data}).json(exclude_none=True),
            headers=headers,
        ).raise_for_status()

    # Action methods below could really use functools.partialmethod. Mypy doesn't like
    # that though. See:
    # -> https://github.com/python/mypy/issues/1484
    # -> https://github.com/python/mypy/issues/2967
    # -> https://github.com/python/mypy/issues/8619
    def adjust(
        self,
        /,
        warehouse_code: str,
        data: Iterable[schema.StockAdjustment],
        session: Optional[str] = None,
    ) -> None:
        self._post_quantity("adjust", warehouse_code, data, session=session)

    def allocate(
        self,
        /,
        warehouse_code: str,
        data: Iterable[schema.Stock],
        session: Optional[str] = None,
    ) -> None:
        self._post_quantity("allocate", warehouse_code, data, session=session)

    def deallocate(
        self,
        /,
        warehouse_code: str,
        data: Iterable[schema.Stock],
        session: Optional[str] = None,
    ) -> None:
        self._post_quantity("deallocate", warehouse_code, data, session=session)

    def receive(
        self,
        /,
        warehouse_code: str,
        data: Iterable[schema.Stock],
        session: Optional[str] = None,
    ) -> None:
        self._post_quantity("receive", warehouse_code, data, session=session)

    def release(
        self,
        /,
        warehouse_code: str,
        data: Iterable[schema.Stock],
        session: Optional[str] = None,
    ) -> None:
        self._post_quantity("release", warehouse_code, data, session=session)
