"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DenyList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const create_map_1 = require("./create-map");
const prune_1 = require("./prune");
/**
 * Manages the construct hub deny list.
 */
class DenyList extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The object key within the bucket with the deny list JSON file.
         */
        this.objectKey = 'deny-list.json';
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            enforceSSL: true,
        });
        const directory = this.writeToFile(props.rules, this.objectKey);
        // upload the deny list to the bucket
        const upload = new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            destinationBucket: this.bucket,
            sources: [s3deploy.Source.asset(directory)],
        });
        this.prune = new prune_1.Prune(this, 'Prune', {
            packageDataBucket: props.packageDataBucket,
            packageDataKeyPrefix: props.packageDataKeyPrefix,
            monitoring: props.monitoring,
        });
        this.grantRead(this.prune.pruneHandler);
        // trigger prune when the deny list changes
        const pruneOnChange = (_a = props.pruneOnChange) !== null && _a !== void 0 ? _a : true;
        if (pruneOnChange) {
            this.prune.pruneHandler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.bucket, {
                events: [s3.EventType.OBJECT_CREATED],
                filters: [{ prefix: this.objectKey, suffix: this.objectKey }],
            }));
        }
        // trigger prune periodically (every 5 minutes) - just in case
        const prunePeriod = (_b = props.prunePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5);
        if (prunePeriod && prunePeriod.toSeconds() > 0) {
            new events.Rule(this, 'PeriodicPrune', {
                schedule: events.Schedule.rate(prunePeriod),
                targets: [new targets.LambdaFunction(this.prune.pruneHandler)],
            });
        }
        // add an explicit dep between upload and the bucket scope which can now
        // also include the bucket notification resource. otherwise, the first
        // upload will not trigger a prune
        upload.node.addDependency(this.bucket);
    }
    /**
     * Grants an AWS Lambda function permissions to read the deny list, and adds
     * the relevant environment variables expected by the `DenyListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment(constants_1.ENV_DENY_LIST_BUCKET_NAME, this.bucket.bucketName);
        handler.addEnvironment(constants_1.ENV_DENY_LIST_OBJECT_KEY, this.objectKey);
        this.bucket.grantRead(handler);
    }
    /**
     * Number of rules in the deny list.
     */
    metricDenyListRules(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "DenyListRuleCount" /* DENY_LIST_RULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * Writes the deny list to a temporary file and returns a path to a directory
     * with the JSON file. The contents of the JSON file is a map where keys are
     * package names (and optionally, version) and the value is the deny list
     * entry. This makes it easier to query by the service.
     *
     * Also performs some validation to make sure there are no duplicate entries.
     *
     * @param list The deny list
     * @returns a path to a temporary directory that can be deployed to S3
     */
    writeToFile(list, fileName) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'deny-list-'));
        const filePath = path.join(tmpdir, fileName);
        const map = create_map_1.createDenyListMap(list);
        fs.writeFileSync(filePath, JSON.stringify(map, null, 2));
        return tmpdir;
    }
}
exports.DenyList = DenyList;
//# sourceMappingURL=data:application/json;base64,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