"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const backend_1 = require("../../../../backend");
const constants_1 = require("../../../../backend/shared/constants");
const monitoring_1 = require("../../../../monitoring");
const catalog_builder_mock_1 = require("./catalog-builder-mock");
const trigger_client_test_1 = require("./trigger.client-test");
const trigger_prune_test_1 = require("./trigger.prune-test");
// we need to pull mock package data from `src/` because we execute in `lib/`
const mockPackageDataDir = path_1.join(__dirname, '..', '..', '..', '..', '..', 'src', '__tests__', 'backend', 'deny-list', 'integ', 'package-data');
const app = new core_1.App();
const stack = new core_1.Stack(app, 'TestDenyList');
const packageData = new s3.Bucket(stack, 'MockDataBucket', {
    autoDeleteObjects: true,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
new aws_s3_deployment_1.BucketDeployment(stack, 'MockData', {
    destinationBucket: packageData,
    sources: [aws_s3_deployment_1.Source.asset(mockPackageDataDir)],
});
const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring');
const denylist = new backend_1.DenyList(stack, 'DenyList', {
    monitoring: monitoring,
    packageDataBucket: packageData,
    packageDataKeyPrefix: constants_1.STORAGE_KEY_PREFIX,
    rules: [
        { package: 'mypackage', reason: '"mypackage" is deprecated' },
        { package: 'your', version: '1.2.3', reason: 'v1.2.3 of "your" has a security issue' },
    ],
});
const catalogBuilderMock = new catalog_builder_mock_1.CatalogBuilderMock(stack, 'CatalogBuilderMock');
denylist.prune.onChangeInvoke(catalogBuilderMock);
const test1 = new trigger_client_test_1.TriggerClientTest(stack, 'ClientTest', {
    invokeAfter: [denylist],
    environment: {
        BUCKET_NAME: denylist.bucket.bucketName,
        FILE_NAME: denylist.objectKey,
    },
});
denylist.grantRead(test1);
const test2 = new trigger_prune_test_1.TriggerPruneTest(stack, 'PruneTest', {
    invokeAfter: [denylist],
    timeout: core_1.Duration.minutes(5),
    environment: {
        BUCKET_NAME: packageData.bucketName,
        TIMEOUT_SEC: core_1.Duration.minutes(2).toSeconds().toString(),
        EXPECTED_KEYS: JSON.stringify([
            'data/your/v1.2.4/world.txt',
            'data/your/v1.2.4/hello.txt',
        ]),
    },
});
packageData.grantRead(test2);
app.synth();
//# sourceMappingURL=data:application/json;base64,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