"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const constants_1 = require("../../../backend/deny-list/constants");
const aws = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
let mockBucketName;
const MOCK_DENY_LIST_BUCKET = 'deny-list-bucket-name';
const MOCK_DENY_LIST_OBJECT = 'my-deny-list.json';
const MOCK_DENY_LIST_MAP = {
    'name/v0.0.0-pre': {
        package: 'name',
        version: '0.0.0-pre',
        reason: 'blocked version',
    },
    '@foo/blocked': {
        package: '@foo/blocked',
        reason: 'block all version of this package please',
    },
};
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME] = MOCK_DENY_LIST_BUCKET;
    process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY] = MOCK_DENY_LIST_OBJECT;
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    delete process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME];
    delete process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY];
    done();
});
test('initial build', () => {
    // GIVEN
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        const denyListResponse = tryMockDenyList(req);
        if (denyListResponse) {
            return cb(null, denyListResponse);
        }
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    // this is the suffix that triggers the catalog builder.
    const docsSuffix = constants.DOCS_KEY_SUFFIX_TYPESCRIPT;
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${docsSuffix}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v0.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v0.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v0.0.0-pre${docsSuffix}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    AWSMock.mock('S3', 'headObject', (req, cb) => {
        const existingKeys = new Set([...mockFirstPage, ...mockSecondPage].map((obj) => obj.Key));
        if (req.Bucket === mockBucketName && existingKeys.has(req.Key)) {
            return cb(null, {});
        }
        class NotFound extends Error {
            constructor() {
                super(...arguments);
                this.code = 'NotFound';
                this.message = 'Not Found';
                this.time = new Date();
            }
        }
        return cb(new NotFound());
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('application/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({
        package: {
            key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.2${constants.ASSEMBLY_KEY_SUFFIX}`,
            versionId: 'VersionID',
        },
    }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
describe('incremental build', () => {
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    const initialScopePackageV2 = {
        description: 'Package @scope/package, version 2.3.4',
        languages: { foo: 'bar' },
        major: 2,
        metadata: npmMetadata,
        name: '@scope/package',
        version: '2.3.4',
    };
    const initialNameV1 = {
        description: 'Package name, version 1.0.0',
        languages: { foo: 'bar' },
        major: 1,
        metadata: npmMetadata,
        name: 'name',
        version: '1.0.0',
    };
    const initialNameV2 = {
        description: 'Package name, version 2.0.0-pre.10',
        languages: { foo: 'bar' },
        major: 2,
        metadata: npmMetadata,
        name: 'name',
        version: '2.0.0-pre.10',
    };
    const initialCatalog = {
        packages: [
            initialScopePackageV2,
            initialNameV1,
            initialNameV2,
        ],
        updatedAt: new Date().toISOString(),
    };
    test('new major version of @scope/package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            const denyListResponse = tryMockDenyList(req);
            if (denyListResponse) {
                return cb(null, denyListResponse);
            }
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '4');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual([
                    initialScopePackageV2,
                    {
                        description: 'Package @scope/package, version 1.2.3',
                        languages: { foo: 'bar' },
                        major: 1,
                        metadata: npmMetadata,
                        name: '@scope/package',
                        version: '1.2.3',
                    },
                    initialNameV1,
                    initialNameV2,
                ]);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('updated un-scoped package version', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            const denyListResponse = tryMockDenyList(req);
            if (denyListResponse) {
                return cb(null, denyListResponse);
            }
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual([
                    initialScopePackageV2,
                    { ...initialNameV1, description: 'Package name, version 1.2.3', version: '1.2.3' },
                    initialNameV2,
                ]);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('ignored "older" minor version of @scope/package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            const denyListResponse = tryMockDenyList(req);
            if (denyListResponse) {
                return cb(null, denyListResponse);
            }
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual(initialCatalog.packages);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('ignored "older" pre-release of package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            const denyListResponse = tryMockDenyList(req);
            if (denyListResponse) {
                return cb(null, denyListResponse);
            }
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual(initialCatalog.packages);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('ignored denied list package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            const denyListResponse = tryMockDenyList(req);
            if (denyListResponse) {
                return cb(null, denyListResponse);
            }
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}@foo/blocked/v1.1.0${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual(initialCatalog.packages);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
function tryMockDenyList(req) {
    if (req.Bucket === MOCK_DENY_LIST_BUCKET && req.Key === MOCK_DENY_LIST_OBJECT) {
        return { Body: JSON.stringify(MOCK_DENY_LIST_MAP) };
    }
    else {
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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