"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.enableStageTransition = exports.disableStageTransition = exports.getStageState = void 0;
const aws_sdk_1 = require("aws-sdk");
const codepipeline = new aws_sdk_1.CodePipeline();
exports.getStageState = async (pipelineName, stageName) => {
    const pipelineState = await codepipeline
        .getPipelineState({ name: pipelineName })
        .promise();
    return pipelineState.stageStates?.find((stage) => stage.stageName === stageName);
};
/**
 * Disables stage transitions. Will only disable if stageState.inboundTransitionState.enabled is false.
 * Strips invalid characters from the reason and truncates to 300 characters.
 */
exports.disableStageTransition = async (params, lastChangedByMustInclude, stageState) => {
    if (stageState?.inboundTransitionState?.enabled === true ||
        (stageState?.inboundTransitionState?.enabled === false &&
            stageState.inboundTransitionState?.lastChangedBy?.includes(lastChangedByMustInclude))) {
        // See documentation https://docs.amazon.com/codepipeline/latest/APIReference/API_DisableStageTransition.html
        params.reason = params.reason
            .replace(/[^a-zA-Z0-9!@ \(\)\.\*\?\-]/g, '-')
            .slice(0, 300);
        await codepipeline.disableStageTransition(params).promise();
    }
};
/**
 * Enables stage transitions. Will only enable if stageState.inboundTransitionState.enabled is true
 * and if lastChangedBy includes the lastChangedByMustInclude string.
 */
exports.enableStageTransition = async (params, lastChangedByMustInclude, stageState) => {
    if (stageState?.inboundTransitionState?.enabled === false &&
        stageState.inboundTransitionState?.lastChangedBy?.includes(lastChangedByMustInclude)) {
        await codepipeline.enableStageTransition(params).promise();
    }
};
//# sourceMappingURL=data:application/json;base64,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