"""
Unit tests for the Rule class.
"""
import unittest
from rulify import Rule


class TestRule(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.condition_true = lambda ctx: True
        self.condition_false = lambda ctx: False
        self.action = lambda ctx: ctx.update({"processed": True}) or ctx
        self.context = {"data": "test"}

    def test_rule_creation(self):
        """Test basic rule creation."""
        rule = Rule("Test Rule", self.condition_true, self.action)
        
        self.assertEqual(rule.name, "Test Rule")
        self.assertEqual(rule.condition, self.condition_true)
        self.assertEqual(rule.action, self.action)
        self.assertEqual(rule.priority, 0)
        self.assertIsNone(rule.next_rule)

    def test_rule_with_priority(self):
        """Test rule creation with custom priority."""
        rule = Rule("Priority Rule", self.condition_true, self.action, priority=10)
        self.assertEqual(rule.priority, 10)

    def test_rule_with_next_rule(self):
        """Test rule creation with chained rule."""
        next_rule = Rule("Next Rule", self.condition_true, self.action)
        rule = Rule("Main Rule", self.condition_true, self.action, next_rule=next_rule)
        
        self.assertEqual(rule.next_rule, next_rule)

    def test_evaluate_condition_true(self):
        """Test rule evaluation when condition is true."""
        rule = Rule("Test Rule", self.condition_true, self.action)
        result = rule.evaluate(self.context.copy())
        
        self.assertIsNotNone(result)
        self.assertTrue(result["processed"])

    def test_evaluate_condition_false(self):
        """Test rule evaluation when condition is false."""
        rule = Rule("Test Rule", self.condition_false, self.action)
        result = rule.evaluate(self.context.copy())
        
        self.assertIsNone(result)

    def test_evaluate_with_chained_rule(self):
        """Test rule evaluation with chained rules."""
        chained_action = lambda ctx: ctx.update({"chained": True}) or ctx
        chained_rule = Rule("Chained Rule", self.condition_true, chained_action)
        
        rule = Rule("Main Rule", self.condition_true, self.action, next_rule=chained_rule)
        result = rule.evaluate(self.context.copy())
        
        self.assertIsNotNone(result)
        self.assertTrue(result["processed"])
        self.assertTrue(result["chained"])

    def test_evaluate_chained_rule_condition_false(self):
        """Test chained rule when main rule condition is false."""
        chained_action = lambda ctx: ctx.update({"chained": True}) or ctx
        chained_rule = Rule("Chained Rule", self.condition_true, chained_action)
        
        rule = Rule("Main Rule", self.condition_false, self.action, next_rule=chained_rule)
        result = rule.evaluate(self.context.copy())
        
        self.assertIsNone(result)

    def test_action_modifies_context(self):
        """Test that action properly modifies the context."""
        def modify_action(ctx):
            ctx["modified"] = True
            ctx["value"] = 42
            return ctx
        
        rule = Rule("Modify Rule", self.condition_true, modify_action)
        context = {"original": "data"}
        result = rule.evaluate(context)
        
        self.assertIsNotNone(result)
        self.assertTrue(result["modified"])
        self.assertEqual(result["value"], 42)
        self.assertEqual(result["original"], "data")


if __name__ == "__main__":
    unittest.main()
