"""
Integration tests for the rulify package.
"""
import unittest
from rulify import Rule, RuleEngine


class TestRulifyIntegration(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.engine = RuleEngine()

    def test_complete_workflow_example(self):
        """Test a complete workflow similar to the README example."""
        def is_hot(ctx):
            return ctx.get("temp", 0) > 30
        
        def turn_on_ac(ctx):
            ctx["ac"] = True
            ctx["fan_speed"] = "high"
            return ctx
        
        def is_cold(ctx):
            return ctx.get("temp", 0) < 20
        
        def turn_on_heater(ctx):
            ctx["heater"] = True
            ctx["temp_setting"] = 25
            return ctx
        
        # Add rules to engine
        self.engine.add_rule(Rule("Hot Check", is_hot, turn_on_ac, priority=10))
        self.engine.add_rule(Rule("Cold Check", is_cold, turn_on_heater, priority=5))
        
        # Test hot weather scenario
        hot_context = {"temp": 35}
        hot_result = self.engine.run(hot_context)
        
        self.assertTrue(hot_result["ac"])
        self.assertEqual(hot_result["fan_speed"], "high")
        self.assertFalse(hot_result.get("heater", False))
        
        # Test cold weather scenario
        cold_context = {"temp": 15}
        cold_result = self.engine.run(cold_context)
        
        self.assertTrue(cold_result["heater"])
        self.assertEqual(cold_result["temp_setting"], 25)
        self.assertFalse(cold_result.get("ac", False))
        
        # Test moderate weather scenario
        moderate_context = {"temp": 25}
        moderate_result = self.engine.run(moderate_context)
        
        self.assertFalse(moderate_result.get("ac", False))
        self.assertFalse(moderate_result.get("heater", False))

    def test_e_commerce_order_processing(self):
        """Test e-commerce order processing workflow."""
        def is_vip_customer(ctx):
            return ctx.get("customer_tier") == "VIP"
        
        def apply_vip_discount(ctx):
            ctx["discount"] = 0.15
            ctx["priority_shipping"] = True
            return ctx
        
        def is_high_value_order(ctx):
            return ctx.get("order_total", 0) > 1000
        
        def apply_bulk_discount(ctx):
            current_discount = ctx.get("discount", 0)
            ctx["discount"] = current_discount + 0.10
            return ctx
        
        def is_express_shipping(ctx):
            return ctx.get("shipping_type") == "express"
        
        def calculate_shipping_cost(ctx):
            base_cost = 10
            if ctx.get("priority_shipping"):
                base_cost += 5
            ctx["shipping_cost"] = base_cost
            return ctx
        
        # Build order processing engine
        self.engine.add_rule(Rule("VIP Customer", is_vip_customer, apply_vip_discount, priority=20))
        self.engine.add_rule(Rule("High Value", is_high_value_order, apply_bulk_discount, priority=15))
        self.engine.add_rule(Rule("Express Shipping", is_express_shipping, calculate_shipping_cost, priority=10))
        
        # Test VIP customer with high value order
        vip_order = {
            "customer_tier": "VIP",
            "order_total": 1200,
            "shipping_type": "express"
        }
        
        result = self.engine.run(vip_order)
        
        self.assertEqual(result["discount"], 0.25)  # 0.15 + 0.10
        self.assertTrue(result["priority_shipping"])
        self.assertEqual(result["shipping_cost"], 15)  # 10 + 5

    def test_rule_chaining_workflow(self):
        """Test complex rule chaining workflow."""
        def check_weather(ctx):
            return ctx.get("weather") == "rainy"
        
        def grab_umbrella(ctx):
            ctx["umbrella"] = True
            return ctx
        
        def check_location(ctx):
            return ctx.get("location") == "outdoor"
        
        def stay_inside(ctx):
            ctx["location"] = "indoor"
            return ctx
        
        def check_time(ctx):
            return ctx.get("time") == "morning"
        
        def grab_coffee(ctx):
            ctx["coffee"] = True
            return ctx
        
        # Create chained rules
        umbrella_rule = Rule("Grab Umbrella", check_weather, grab_umbrella)
        coffee_rule = Rule("Grab Coffee", check_time, grab_coffee, next_rule=umbrella_rule)
        location_rule = Rule("Stay Inside", check_location, stay_inside, next_rule=coffee_rule)
        
        self.engine.add_rule(location_rule)
        
        # Test outdoor, morning, rainy scenario
        context = {
            "weather": "rainy",
            "location": "outdoor",
            "time": "morning"
        }
        
        result = self.engine.run(context)
        
        self.assertEqual(result["location"], "indoor")
        self.assertTrue(result["coffee"])
        self.assertTrue(result["umbrella"])

    def test_error_handling_in_conditions(self):
        """Test error handling in rule conditions."""
        def safe_condition(ctx):
            try:
                return ctx["nonexistent_key"] > 0
            except KeyError:
                return False
        
        def safe_action(ctx):
            ctx["safe_processed"] = True
            return ctx
        
        rule = Rule("Safe Rule", safe_condition, safe_action)
        self.engine.add_rule(rule)
        
        context = {"data": "test"}
        result = self.engine.run(context)
        
        # Should not crash and should not process
        self.assertFalse(result.get("safe_processed", False))

    def test_action_error_handling(self):
        """Test error handling in rule actions."""
        def condition(ctx):
            return True
        
        def error_action(ctx):
            raise ValueError("Test error")
        
        def safe_action(ctx):
            ctx["safe_processed"] = True
            return ctx
        
        # Add rules
        self.engine.add_rule(Rule("Error Rule", condition, error_action, priority=10))
        self.engine.add_rule(Rule("Safe Rule", condition, safe_action, priority=5))
        
        context = {"data": "test"}
        
        # This should raise a RuleEngineError
        from rulify.exceptions import RuleEngineError
        with self.assertRaises(RuleEngineError):
            self.engine.run(context)


if __name__ == "__main__":
    unittest.main()
