"""
Unit tests for the RuleEngine class.
"""
import unittest
from rulify import Rule, RuleEngine


class TestRuleEngine(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.engine = RuleEngine()
        self.context = {"data": "test"}

    def test_engine_creation(self):
        """Test basic engine creation."""
        self.assertEqual(len(self.engine.rules), 0)

    def test_add_rule(self):
        """Test adding a rule to the engine."""
        rule = Rule("Test Rule", lambda ctx: True, lambda ctx: ctx)
        result = self.engine.add_rule(rule)
        
        self.assertEqual(len(self.engine.rules), 1)
        self.assertEqual(self.engine.rules[0], rule)
        self.assertEqual(result, self.engine)  # Should return self for chaining

    def test_add_multiple_rules(self):
        """Test adding multiple rules to the engine."""
        rule1 = Rule("Rule 1", lambda ctx: True, lambda ctx: ctx, priority=1)
        rule2 = Rule("Rule 2", lambda ctx: True, lambda ctx: ctx, priority=2)
        rule3 = Rule("Rule 3", lambda ctx: True, lambda ctx: ctx, priority=3)
        
        self.engine.add_rule(rule1)
        self.engine.add_rule(rule2)
        self.engine.add_rule(rule3)
        
        self.assertEqual(len(self.engine.rules), 3)
        # Rules should be sorted by priority (highest first)
        self.assertEqual(self.engine.rules[0], rule3)
        self.assertEqual(self.engine.rules[1], rule2)
        self.assertEqual(self.engine.rules[2], rule1)

    def test_run_with_no_rules(self):
        """Test running engine with no rules."""
        result = self.engine.run(self.context.copy())
        self.assertEqual(result, self.context)

    def test_run_with_single_rule_condition_true(self):
        """Test running engine with single rule that matches."""
        def action(ctx):
            ctx["processed"] = True
            return ctx
        
        rule = Rule("Test Rule", lambda ctx: True, action)
        self.engine.add_rule(rule)
        
        result = self.engine.run(self.context.copy())
        self.assertTrue(result["processed"])

    def test_run_with_single_rule_condition_false(self):
        """Test running engine with single rule that doesn't match."""
        def action(ctx):
            ctx["processed"] = True
            return ctx
        
        rule = Rule("Test Rule", lambda ctx: False, action)
        self.engine.add_rule(rule)
        
        result = self.engine.run(self.context.copy())
        self.assertFalse(result.get("processed", False))

    def test_run_with_multiple_rules(self):
        """Test running engine with multiple rules."""
        def action1(ctx):
            ctx["rule1"] = True
            return ctx
        
        def action2(ctx):
            ctx["rule2"] = True
            return ctx
        
        rule1 = Rule("Rule 1", lambda ctx: True, action1, priority=1)
        rule2 = Rule("Rule 2", lambda ctx: True, action2, priority=2)
        
        self.engine.add_rule(rule1)
        self.engine.add_rule(rule2)
        
        result = self.engine.run(self.context.copy())
        self.assertTrue(result["rule1"])
        self.assertTrue(result["rule2"])

    def test_run_with_mixed_conditions(self):
        """Test running engine with rules that have different conditions."""
        def action1(ctx):
            ctx["rule1"] = True
            return ctx
        
        def action2(ctx):
            ctx["rule2"] = True
            return ctx
        
        rule1 = Rule("Rule 1", lambda ctx: True, action1, priority=1)
        rule2 = Rule("Rule 2", lambda ctx: False, action2, priority=2)
        
        self.engine.add_rule(rule1)
        self.engine.add_rule(rule2)
        
        result = self.engine.run(self.context.copy())
        self.assertTrue(result["rule1"])
        self.assertFalse(result.get("rule2", False))

    def test_run_preserves_context_order(self):
        """Test that context modifications are preserved in order."""
        def action1(ctx):
            ctx["step"] = 1
            return ctx
        
        def action2(ctx):
            ctx["step"] = 2
            return ctx
        
        rule1 = Rule("Rule 1", lambda ctx: True, action1, priority=1)
        rule2 = Rule("Rule 2", lambda ctx: True, action2, priority=2)
        
        self.engine.add_rule(rule1)
        self.engine.add_rule(rule2)
        
        result = self.engine.run(self.context.copy())
        # Rule2 should run first (higher priority), then rule1
        self.assertEqual(result["step"], 1)

    def test_run_with_chained_rules(self):
        """Test running engine with chained rules."""
        def action1(ctx):
            ctx["step1"] = True
            return ctx
        
        def action2(ctx):
            ctx["step2"] = True
            return ctx
        
        chained_rule = Rule("Chained Rule", lambda ctx: True, action2)
        main_rule = Rule("Main Rule", lambda ctx: True, action1, next_rule=chained_rule)
        
        self.engine.add_rule(main_rule)
        result = self.engine.run(self.context.copy())
        
        self.assertTrue(result["step1"])
        self.assertTrue(result["step2"])

    def test_rule_priority_sorting(self):
        """Test that rules are properly sorted by priority."""
        rules = []
        for i in range(5):
            rule = Rule(f"Rule {i}", lambda ctx: True, lambda ctx: ctx, priority=i)
            rules.append(rule)
            self.engine.add_rule(rule)
        
        # Rules should be sorted by priority (highest first)
        for i, rule in enumerate(self.engine.rules):
            self.assertEqual(rule.priority, 4 - i)

    def test_add_rule_chaining(self):
        """Test that add_rule returns self for method chaining."""
        rule1 = Rule("Rule 1", lambda ctx: True, lambda ctx: ctx)
        rule2 = Rule("Rule 2", lambda ctx: True, lambda ctx: ctx)
        
        result = self.engine.add_rule(rule1).add_rule(rule2)
        
        self.assertEqual(result, self.engine)
        self.assertEqual(len(self.engine.rules), 2)


if __name__ == "__main__":
    unittest.main()
