"""
Rule implementation for the rulify package.
"""
from .exceptions import RuleConditionError, RuleActionError


class Rule:
    """
    Represents a single rule with condition, action, and metadata.
    """
    
    def __init__(self, name, condition, action, priority=0, next_rule=None):
        """
        Initialize a rule.
        
        Args:
            name (str): Human-readable name for the rule
            condition (callable): Function that returns True/False based on context
            action (callable): Function to execute when condition is True
            priority (int): Higher numbers execute first (default: 0)
            next_rule (Rule): Optional rule to chain after this one
            
        Raises:
            ValueError: If name is empty or None
            TypeError: If condition or action are not callable
        """
        if not name or not isinstance(name, str):
            raise ValueError("Rule name must be a non-empty string")
        
        if not callable(condition):
            raise TypeError("Condition must be callable")
        
        if not callable(action):
            raise TypeError("Action must be callable")
        
        self.name = name
        self.condition = condition
        self.action = action
        self.priority = priority
        self.next_rule = next_rule

    def evaluate(self, context):
        """
        Evaluate the rule against the given context.
        
        Args:
            context: Dictionary or object to evaluate against
            
        Returns:
            The result of the action if condition is True, None otherwise
            
        Raises:
            RuleConditionError: If condition evaluation fails
            RuleActionError: If action execution fails
        """
        try:
            if self.condition(context):
                try:
                    result = self.action(context)
                    if self.next_rule:
                        return self.next_rule.evaluate(result)
                    return result
                except Exception as e:
                    raise RuleActionError(f"Action '{self.name}' failed: {str(e)}") from e
        except Exception as e:
            raise RuleConditionError(f"Condition '{self.name}' failed: {str(e)}") from e
        
        return None
