"""
Rule engine implementation for the rulify package.
"""
from .exceptions import RuleEngineError


class RuleEngine:
    """
    A rule engine that manages and executes rules in priority order.
    """
    
    def __init__(self):
        """Initialize the rule engine."""
        self.rules = []

    def add_rule(self, rule):
        """
        Add a rule to the engine.
        
        Args:
            rule: A Rule instance to add to the engine
            
        Returns:
            self: For method chaining
            
        Raises:
            RuleEngineError: If the rule is not a valid Rule instance
        """
        if not hasattr(rule, 'evaluate'):
            raise RuleEngineError("Rule must have an 'evaluate' method")
        
        self.rules.append(rule)
        self.rules.sort(key=lambda r: r.priority, reverse=True)
        return self

    def run(self, context):
        """
        Execute all rules against the given context.
        
        Args:
            context: Dictionary or object to process through the rules
            
        Returns:
            The processed context after all applicable rules have been executed
            
        Raises:
            RuleEngineError: If context processing fails
        """
        if context is None:
            raise RuleEngineError("Context cannot be None")
        
        try:
            for rule in self.rules:
                result = rule.evaluate(context)
                if result is not None:
                    context = result
            return context
        except Exception as e:
            raise RuleEngineError(f"Error executing rules: {str(e)}") from e
