"""PyTrickle CLI: scaffold starter apps for building streaming pipelines.

Usage:
  pytrickle init my_app --port 8000 --out ./my_app.py [--force]

This generates a template based on examples/passthrough_example.py.
The template is a real, validated Python file that can also be run directly.
"""

from __future__ import annotations

import argparse
import ast
import os
import re
from pathlib import Path
from textwrap import dedent

from importlib.resources import files


def _get_examples_dir() -> Path:
    """Resolve the examples directory location.

    Tries importlib.resources first (installed package) and then local filesystem
    (editable/development installs).
    """
    try:
        if files is not None:
            # Get the path from importlib.resources
            # Use "pytrickle.examples" since that's our package name (mapped to root examples/)
            examples_path = files("pytrickle.examples")
            # Convert to Path - handling different return types
            if hasattr(examples_path, '__fspath__'):
                return Path(examples_path)
            return Path(str(examples_path))
    except (ModuleNotFoundError, AttributeError, TypeError):
        pass
    
    # Fallback: examples dir at repo root (works in development mode)
    # cli.py is in pytrickle/, so parent.parent gets us to repo root
    examples_dir = Path(__file__).parent.parent / "examples"
    
    if examples_dir.exists() and examples_dir.is_dir():
        return examples_dir
    
    raise FileNotFoundError(
        "examples directory not found; install pytrickle or run from source"
    )


def _list_templates() -> dict[str, str]:
    """Return mapping of template name -> file path from examples."""
    examples_dir = _get_examples_dir()
    templates = {}
    
    # Find all Python files except __init__.py
    for file_path in examples_dir.glob("*.py"):
        if file_path.name != "__init__.py":
            # Extract template name (remove _example.py suffix if present)
            template_name = file_path.stem.replace("_example", "")
            templates[template_name] = str(file_path)
    
    return templates


def _get_template(template_name: str) -> str:
    """Load template content by name (accepts both foo and foo_example)."""
    templates = _list_templates()
    
    # Try exact match first
    if template_name in templates:
        return Path(templates[template_name]).read_text()
    
    # Try with _example suffix
    alt_name = f"{template_name}_example"
    if alt_name in templates:
        return Path(templates[alt_name]).read_text()
    
    # List available templates
    available = ", ".join(sorted(templates.keys()))
    raise ValueError(
        f"Template '{template_name}' not found. Available templates: {available}"
    )


def _customize_template(template: str, app_name: str, port: int, output_file: str) -> str:
    """Apply simple substitutions to produce a runnable starter file."""
    # Update the name in StreamProcessor.from_handlers (match any name)
    template = re.sub(
        r'name="[^"]*"',
        f'name="{app_name}"',
        template,
        count=1
    )
    
    # Update the port (first occurrence in StreamProcessor.from_handlers)
    template = re.sub(
        r'port=\d+',
        f'port={port}',
        template,
        count=1
    )
    
    # Annotate the docstring with generation info
    template = re.sub(
        r'("""[^"]+?)\n\n',
        f'\\1\n\nGenerated by: pytrickle init {app_name}\n\n',
        template,
        count=1
    )
    
    # Update the run command in docstring (any example name)
    template = re.sub(
        r'python -m pytrickle\.examples\.\w+',
        f'python {output_file}',
        template,
    )
    
    # Drop scaffold instructions from the source example if present
    template = re.sub(
        r'\n\nOr generate a customized copy:.*?pytrickle init my_app.*?\n',
        '\n',
        template,
        flags=re.DOTALL
    )
    
    return template


def _validate_python_syntax(code: str) -> None:
    """Raise if the generated file is not valid Python."""
    try:
        ast.parse(code)
    except SyntaxError as e:
        raise ValueError(f"Generated code has syntax error at line {e.lineno}: {e.msg}")


def _write_file(path: Path, content: str, force: bool = False) -> None:
    """Write content to disk; guard against accidental overwrite unless forced."""
    if path.exists() and not force:
        raise FileExistsError(
            f"File already exists: {path}\nUse --force to overwrite"
        )
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(content, encoding="utf-8")


def cmd_list(args: argparse.Namespace) -> None:
    """Print available templates with a short summary from their docstrings."""
    try:
        templates = _list_templates()
        
        print("Available templates:")
        print()
        
        # Read first docstring from each template for description
        for name in sorted(templates.keys()):
            template_path = Path(templates[name])
            try:
                content = template_path.read_text()
                # Extract first line of docstring
                match = re.search(r'"""(.+?)\.', content, re.DOTALL)
                if match:
                    desc = match.group(1).strip().replace('\n', ' ')
                    # Limit description length
                    if len(desc) > 60:
                        desc = desc[:57] + "..."
                else:
                    desc = "No description available"
            except Exception:
                desc = "Could not read template"
            
            print(f"  \033[92m{name:20}\033[0m {desc}")
        
        print()
        print("Usage: pytrickle init <app_name> --template <template_name>")
        
    except Exception as e:
        print(f"\033[91m Error listing templates: {e}\033[0m")
        raise


def cmd_init(args: argparse.Namespace) -> None:
    """Create a new app file from a chosen template."""
    app_name = args.name
    port = int(args.port)
    out = Path(args.out)
    template_name = args.template
    
    # If output is a directory, append the app name as filename
    if out.is_dir():
        out = out / f"{app_name}.py"
    
    try:
        # Read the specified template
        template = _get_template(template_name)
        
        # Customize it
        customized = _customize_template(template, app_name, port, os.fspath(out))
        
        # Validate the output
        _validate_python_syntax(customized)
        
        # Write the file
        _write_file(out, customized, force=bool(args.force))
        
        # Success message
        print(f"✅ Created: {out}")
        print(f"   Template: {template_name}")
        print(f"   Run with: \033[92mpython {out}\033[0m")
        print("   Edit the handlers to customize your pipeline.")
        
    except FileExistsError as e:
        print(f"\033[91m Error: {e}\033[0m")
        return
    except ValueError as e:
        print(f"\033[91m {e}\033[0m")
        return
    except Exception as e:
        print(f"\033[91m Unexpected error: {e}\033[0m")
        raise


def build_parser() -> argparse.ArgumentParser:
    """Configure the CLI interface."""
    p = argparse.ArgumentParser(
        prog="pytrickle",
        description="Scaffold streaming pipeline starter apps",
        epilog=dedent(
            """
            Examples:
              # List available templates
              pytrickle list
              
              # Create app from default template
              pytrickle init my_app
              
              # Create app from specific template
              pytrickle init my_app --template grayscale_chipmunk
              pytrickle init my_app --template process_video --port 9000
              
              # Create in specific directory
              pytrickle init my_app --out ./apps/ --force
            
            Templates come from pytrickle.examples and are runnable as-is
            (e.g., python -m pytrickle.examples.passthrough).
            """
        ),
        formatter_class=argparse.RawTextHelpFormatter,
    )
    sub = p.add_subparsers(dest="cmd", required=True)

    # List command
    list_parser = sub.add_parser(
        "list",
        help="List all available templates"
    )
    list_parser.set_defaults(func=cmd_list)

    # Init command
    init_parser = sub.add_parser(
        "init",
        help="Create a starter app from template"
    )
    init_parser.add_argument(
        "name",
        help="App name (used in service name and filename if out is a directory)"
    )
    init_parser.add_argument(
        "--template",
        "-t",
        default="passthrough",
        help="Template to use (default: passthrough). Use 'pytrickle list' to see all templates."
    )
    init_parser.add_argument(
        "--port",
        type=int,
        default=8000,
        help="Port to bind the server (default: 8000)"
    )
    init_parser.add_argument(
        "--out",
        default="./",
        help="Output file or directory (default: current directory)"
    )
    init_parser.add_argument(
        "--force",
        action="store_true",
        help="Overwrite target if exists"
    )
    init_parser.set_defaults(func=cmd_init)

    return p


def main(argv: list[str] | None = None) -> None:
    parser = build_parser()
    args = parser.parse_args(argv)
    args.func(args)


if __name__ == "__main__":  # pragma: no cover
    main()
