"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.8.2" };
//# sourceMappingURL=data:application/json;base64,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