# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bevy']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'bevy',
    'version': '0.4.7',
    'description': 'Python Dependency Inversion made simple so you can focus on creating amazing code.',
    'long_description': '# Bevy\nBevy makes using *Dependency Injection* a breeze so that you can focus on creating amazing code.\n\n## Installation\n```shell script\npip install bevy\n```\n\n**[Documentation](docs/documentation.md)**\n\n## Dependency Injection\nPut simply, *Dependency Injection* is a design pattern where the objects that your class depends on are instantiated outside of the class. Those dependencies are then injected into your class when it is instantiated.\nThis promotes loosely coupled code where your class doesn’t require direct knowledge of what classes it depends on or how to create them. Instead your class declares what class interface it expects and an outside framework handles the work of creating the class instances with the correct interface.\n## Interfaces\nPython doesn’t have an actual interface implementation like many other languages. Class inheritance, however, can be used in a very similar way since sub classes will likely have the same fundamental interface as their base class. \n## Why Do I Care?\n*Dependency Injection* and its reliance on abstract interfaces makes your code easier to maintain:\n- Changes can be made without needing to alter implementation details in unrelated code, so long as the interface isn’t modified in a substantial way.\n- Tests can provide mock implementations of dependencies without needing to jump through hoops to inject them. They can provide the mock to the context and Bevy will make sure it is used where appropriate.\n## How Bevy Works\nBevy allows your class to say what dependencies it has by using undefined class attribute annotations. That’s a bit vague so here is an example:\n```py\nclass MyClass(Injectable):\n    my_dependency: MyDependency\n```\nThe class `MyClass`has a dependency on the `MyDependency` interface. It wants this dependency to be made available with the attribute name `my_dependency`. This will allow your class to access an instance of `MyDependency`as `self.my_dependency`, even from `__init__` since the injection happens when `__new__` is called.\n\nIt is important to note that Bevy ignores any class attribute that has been assigned to. For example:\n```py\nmy_dependency: MyDependency = MyDependency()\n```\nThis will be ignored by the dependency resolver because it’s been assigned a value.\n## Dependency Resolution\nDependency resolution and injection is handled when `__new__` is called. Bevy keeps a repository of all dependencies that have already been created in a context. This repository is used to look for each of the dependency class interfaces when a Bevy `Injectable` class is created. A dependency will be used from the repository only if it is the same class as the interface requested or a sub-class of that interface. If no match is found for an interface, an instance will be created without arguments and saved to the repository.\n\nSo, in short, all dependencies are guaranteed to be either the same class as the dependency’s interface or a sub class of that interface.\n## How To Customize Dependencies\nIf you need to instantiate a dependency with arguments or provide an alternate implementation of a dependency you can create a custom context.\n\n```py\nfrom bevy import Context\n\ncontext = Context().add(MyDependency(foo="bar"))\napp = context.create(MyApp, "some instantiation args")\n```\nIt is important to note that `Context.create` does not add the instance returned to the context repository. If that is necessary use `Context.get`, or if you need to pass instantiation arguments use `Context.load` passing the instance returned by `Context.create`.\n## Dependency Factories\nIt is also possible to create a factory for any of your dependencies. Instances generated by a factory will not be added to the context repository since they will not be unique in the context. Creating a factory is as simple as annotating a class attribute with the bevy factory class and telling it what dependency type it should create.\n\n```py\nfrom bevy import Injectable, Factory\n\n\nclass MyApp(Injectable):\n    factory: Factory[My_Dependency]\n\n    def get_instance(self, name: str) -> My_Dependency:\n        return self.factory(name)\n```\n## Accessing The Context\nYou can give a Bevy `Injectable` object access to the context that created it by adding a class attribute annotated with the `Context` type. This will cause the current context instance to inject itself into your class.\n\n```py\nfrom bevy import Injectable, Context\n\n\nclass MyApp(Injectable):\n    context: Context\n```\n## Scoped Contexts\nIt is possible to branch a context to create a child context which has access to everything in its repository and in the repositories of its parent contexts, while the parents do not have access to the repository of the child. This might be used for a plugin system where you’d want the plugin to have access to the dependencies of the app but you wouldn’t want the plugin to pollute the app’s context.\n```py\nclass MyApp(Injectable):\n    context: Context\n\n    def __init__(self, plugins: List[Type[Plugin]]):\n        self.plugins = self.load_plugins(plugins)\n\n    def load_plugins(self, plugins: List[Type[Plugin]]) -> List[Plugin]:\n        plugin_instances = []\n        for plugin in plugins:\n            instance = self.context.branch().create(plugin)\n            plugin_instances.append(instance)\n        return plugin_instances\n```\n## Injector Dependencies\n\nA dependency class object can implement the bevy.injector.InjectorProtocol by defining a __bevy_inject__ classmethod.\nThe context manager will call this method when it is injecting dependencies into a new instance of an injectable class.\nThe __bevy_inject__ method will be passed the context, the partial instance being built, and all arguments passed to the\nconstructor. It should return an instance of the injector dependency class.\n```py\nclass MyInjector:\n    def __bevy_injector__(self, context: bevy.Context, instance, *args, **kwargs):\n        return MyInjector()\n    ...\n```\n\n## Driving Motivations\nThe motivations that drive the decisions about how Bevy is implemented are as follows.\n- It should feel like nothing has been changed from normal.\n- IDEs should be able to understand what is happening.\n- Everything should work independently.\n\n## Future\n- Add support for circular dependencies. Likely use descriptors to lazily inject dependencies.\n- Add more tests that cover cases beyond the main use cases.\n',
    'author': 'Zech Zimmerman',
    'author_email': 'hi@zech.codes',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ZechCodes/Bevy',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
