import unittest
import datetime

import django
from django.conf import settings

settings.configure(**{
    'ALLOWED_HOSTS': ['testserver'],
    })

django.setup()

from django.db import models
from elasticsearch_dsl import String
import springy


class MyModel(models.Model):
    test_field = models.CharField(max_length=100)
    class Meta:
        app_label = 'test'


class RelatedModel(models.Model):
    test_related_field = models.CharField(max_length=100)
    class Meta:
        app_label = 'test'


class WithRelatedFieldModel(models.Model):
    test_field = models.CharField(max_length=100)
    related = models.ForeignKey(RelatedModel)

    class Meta:
        app_label = 'test'


class PreparingDocTypeTestCase(unittest.TestCase):
    def setUp(self):
        self.obj = MyModel(test_field='test value')

        class SpecialFieldAttributeErrorIndex(springy.Index):
            special_field = String()

            class Meta:
                fields = ('test_field', 'special_field',)
                model = MyModel
                index = 'special'

            def prepare_special_field(self, obj):
                raise AttributeError('Inner attribute error')

        class SimpleTestIndex(springy.Index):
            special_field = String()

            class Meta:
                fields = ('test_field', 'special_field',)
                model = MyModel
                index = 'simple'

            def prepare_special_field(self, obj):
                return 'special value'

        self.simple_idx = SimpleTestIndex()
        self.special_field_idx = SpecialFieldAttributeErrorIndex()

    def tearDown(self):
        springy.registry.unregister_all()

    def test_that_custom_prepare_field_func_raise_inner_attribute_error(self):
        with self.assertRaises(AttributeError):
            self.special_field_idx.to_doctype(self.obj)

    def test_that_doctype_preparation_contains_model_field_values(self):
        dt = self.simple_idx.to_doctype(self.obj)
        self.assertEqual(dt.test_field, 'test value')

    def test_that_doctype_preparation_contains_model_prepared_values(self):
        dt = self.simple_idx.to_doctype(self.obj)
        self.assertEqual(dt.special_field, 'special value')


class IndexDefinitionTestCase(unittest.TestCase):
    def test_that_using_undefined_field_raises_an_exception(self):
        with self.assertRaises(springy.exceptions.FieldDoesNotExist):
            class MissingFieldTestIndex(springy.Index):
                class Meta:
                    fields = ('test_field', 'undefined_field',)
                    model = MyModel
                    index = 'undefined'


class IndexRegistryTestCase(unittest.TestCase):
    def tearDown(self):
        springy.registry.unregister_all()

    def test_successful_autoregistering_index(self):
        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel
                index = 'index'

        self.assertEqual(len(springy.registry.get_all()), 1)
        self.assertEqual(springy.registry.get_all()[0], MyTestIndex)

    def test_successful_registering_index_at_given_name(self):
        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel
                index = 'my_index'

        self.assertEqual(springy.registry.get('my_index'), MyTestIndex)

    def test_successful_registering_index_with_autogenerated_name(self):
        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel

        self.assertEqual(springy.registry.get('tests.test_indices_MyTestIndex'), MyTestIndex)

    def test_successul_unregistering_by_class(self):
        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel

        springy.registry.unregister(MyTestIndex)
        self.assertEqual(len(springy.registry.get_all()), 0)

    def test_raise_exception_when_unregistering_unregistered_index(self):
        from springy.indices import NotRegisteredError

        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel
        springy.registry.unregister_all()

        with self.assertRaises(NotRegisteredError):
            springy.registry.unregister(MyTestIndex)

    def test_raise_exception_when_registering_index_again(self):
        from springy.indices import AlreadyRegisteredError

        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel
                index = 'index'

        with self.assertRaises(AlreadyRegisteredError):
            springy.registry.register('index', MyTestIndex)

    def test_successful_registering_same_index_class_with_other_name(self):
        from springy.indices import AlreadyRegisteredError

        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel

        springy.registry.register('custom_name', MyTestIndex)
        self.assertEqual(len(springy.registry.get_all()), 2)

    def test_successful_unregistering_many_indices_by_class(self):
        class MyTestIndex(springy.Index):
            class Meta:
                fields = ('test_field',)
                model = MyModel
        springy.registry.register('custom_name', MyTestIndex)

        springy.registry.unregister(MyTestIndex)
        self.assertEqual(len(springy.registry.get_all()), 0)

