import unittest
from datetime import datetime
from pathlib import Path

from borb.io.read.types import Decimal
from borb.pdf.canvas.color.color import HexColor, X11Color
from borb.pdf.canvas.geometry.rectangle import Rectangle
from borb.pdf.canvas.layout.layout_element import Alignment
from borb.pdf.canvas.layout.page_layout.multi_column_layout import SingleColumnLayout
from borb.pdf.canvas.layout.table.fixed_column_width_table import (
    FixedColumnWidthTable as Table,
)
from borb.pdf.canvas.layout.text.paragraph import Paragraph
from borb.pdf.document import Document
from borb.pdf.page.page import Page
from borb.pdf.pdf import PDF


class TestWriteParagraphJustifiedCenter(unittest.TestCase):
    """
    This test creates a PDF with a Paragraph object in it. The Paragraph is aligned BOTTOM, CENTERED.
    """

    def __init__(self, methodName="runTest"):
        super().__init__(methodName)
        # find output dir
        p: Path = Path(__file__).parent
        while "output" not in [x.stem for x in p.iterdir() if x.is_dir()]:
            p = p.parent
        p = p / "output"
        self.output_dir = Path(p, Path(__file__).stem.replace(".py", ""))
        if not self.output_dir.exists():
            self.output_dir.mkdir()

    def test_write_document_001(self):
        doc: Document = Document()
        page: Page = Page()
        doc.append_page(page)

        # add test information
        layout = SingleColumnLayout(page)
        layout.add(
            Table(number_of_columns=2, number_of_rows=3)
            .add(Paragraph("Date", font="Helvetica-Bold"))
            .add(Paragraph(datetime.now().strftime("%d/%m/%Y, %H:%M:%S")))
            .add(Paragraph("Test", font="Helvetica-Bold"))
            .add(Paragraph(Path(__file__).stem))
            .add(Paragraph("Description", font="Helvetica-Bold"))
            .add(
                Paragraph(
                    "This test creates a PDF with a Paragraph object in it. The Paragraph is aligned TOP, CENTERED."
                )
            )
            .set_padding_on_all_cells(Decimal(2), Decimal(2), Decimal(2), Decimal(2))
        )

        p: Paragraph = Paragraph(
            "Hello World!", horizontal_alignment=Alignment.CENTERED
        )

        # the next line of code uses absolute positioning
        r: Rectangle = Rectangle(
            Decimal(59),  # x: 0 + page_margin
            Decimal(
                848 - 84 - 200 - 100
            ),  # y: page_height - page_margin - y - height_of_textbox
            Decimal(595 - 59 * 2),  # width: page_width - 2 * page_margin
            Decimal(100),
        )  # height

        # this is a quick and dirty way to draw a rectangle on the page
        page.append_square_annotation(r, stroke_color=X11Color("Red"))

        # add the paragraph to the page
        p.layout(page, r)

        # determine output location
        out_file = self.output_dir / "output_001.pdf"
        with open(out_file, "wb") as out_file_handle:
            PDF.dumps(out_file_handle, doc)

    def test_write_document_002(self):

        # create output directory if it does not exist yet
        if not self.output_dir.exists():
            self.output_dir.mkdir()

        # create document
        pdf = Document()

        # add page
        page = Page()
        pdf.append_page(page)

        # add test information
        layout = SingleColumnLayout(page)
        layout.add(
            Table(number_of_columns=2, number_of_rows=3)
            .add(Paragraph("Date", font="Helvetica-Bold"))
            .add(Paragraph(datetime.now().strftime("%d/%m/%Y, %H:%M:%S")))
            .add(Paragraph("Test", font="Helvetica-Bold"))
            .add(Paragraph(Path(__file__).stem))
            .add(Paragraph("Description", font="Helvetica-Bold"))
            .add(
                Paragraph(
                    "This test creates a PDF with a Paragraph object in it. The Paragraph is aligned BOTTOM, CENTERED."
                )
            )
            .set_padding_on_all_cells(Decimal(2), Decimal(2), Decimal(2), Decimal(2))
        )

        Paragraph(
            """
            Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. 
            Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. 
            Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. 
            Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.
            """,
            font_size=Decimal(12),
            text_alignment=Alignment.CENTERED,
            vertical_alignment=Alignment.BOTTOM,
        ).layout(
            page,
            Rectangle(Decimal(59), Decimal(550), Decimal(476), Decimal(124)),
        )

        # add rectangle annotation
        page.append_square_annotation(
            stroke_color=HexColor("72A276"),
            rectangle=Rectangle(Decimal(59), Decimal(550), Decimal(476), Decimal(124)),
        )

        # determine output location
        out_file = self.output_dir / "output_002.pdf"

        # attempt to store PDF
        with open(out_file, "wb") as in_file_handle:
            PDF.dumps(in_file_handle, pdf)
