# -*- coding: utf-8 -*-
"""varsSetter.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1RF1P_uV7CThARGZJQvSH087nE8QgzTEz
"""

# -----------------------------------------------------------
# Object containing methods to manage the parameters of the model.
#
# (C) 2021 Borja García, Valladolid, Spain
# Released under MIT License (MIT)
# email borjagarmarpy@gmail.com
# -----------------------------------------------------------

import sys
sys.path.append('/content/drive/MyDrive/Burbuja/Difusion-Trabajando/Metodos')
import mFits as ms
import math


class vars():

  """
  Object containing parameters managment of parameters of the model.

  The method is the hierarchical parent of the main model, allowing parameter managment
  to it. Setter methods with no further description consist on a simple structure allowing to read
  a numeric value or setting a default value.
  
  """

  delta_b = None
  delta_s = None
  
  def setD( self, D = 2.2e-9 ):
    """
    Method used to set D (gas diffusion in the liquid) parameter.
    The default value is 2.2e-9 m2/s, the value of diffusion for oxygen in water.
    """
    if( type(D) == int or
        type(D) == float ):
      self.D = D
    else:
      raise NumberCheckError('Diffusivity (D)')

  def setM( self, m = 3.2e-2 ):
    """
    Method used to set m (adimensional Henry's constant) parameter.
    The default value is 3.2e-2, the value for oxygen at 298.15 K.
    """
    if( type(m) == int or
        type(m) == float ):
      self.m = m
    else:
      raise NumberCheckError('Adimensional Henry\'s constant (m)')

  def setT( self, T = 25 ):
    """
    Method used to set T (temperature) parameter.
    Default value is 25ºC, the usual value for room temperature.
    """
    if( type(T) == int or
        type(T) == float ):
      self.T = T
    else:
      raise NumberCheckError('Temperature (T)')

  def setRC( self, rc = 5e-3 ): # m - ancho tubería
    """
    Method used to set rc (radius of the tube) parameter.
    """
    if( type(rc) == int or
        type(rc) == float ):
      self.rc = rc   
    else:
      raise NumberCheckError('Tube Radius (rc)')

  def setVisc( self, visc = None ):
    """
    Method used to set the viscosity parameter.
    The defalt value of the parameter is set by interpolating experimental data.
    """
    if( type(visc) == type(None) ):
      self.visc = ms.viscosity( temperature= self.T )
    else:
      if( type(visc) == int or
          type(visc) == float ):
        self.visc = visc
      else:
        raise NumberCheckError('Viscosity (visc)')

  def setSurfTens( self, surften = None ):
    """
    Method used to set the surface tension parameter.
    The defalt value of the parameter is set by interpolating experimental data.
    """
    if( type(surften) == type(None) ):
      self.surften = ms.surface_tension( temperature = self.T )
    else:
      if( type(surften) == int or
          type(surften) == float ):
        self.surften = surften
      else:
        raise NumberCheckError('Surface Tension (surften)')

  def setDens( self, dens = None ):
    """
    Method used to set the density parameter.
    The defalt value of the parameter is set by interpolating experimental data.
    """
    if( type(dens) == type(None) ):
      self.dens = ms.density( temperature = self.T )
    else:
      if( type(dens) == int or
          type(dens) == float ):
        self.dens = dens
      else:
        raise NumberCheckError('Density (dens)')

  def setZmax( self, zmax = 5 ):
    """
    Method used to set the zmax (maximum in z-range) parameter.
    """
    if( type(zmax) == int or
        type(zmax) == float):
      self.zmax = zmax
    else:
      raise NumberCheckError('maximum in z-axis (zmax)')

  def setZmin( self, zmin = 0.01 ):
    """
    Method used to set the zmin (minimum in z-range) parameter.
    """
    if( type(zmin) == int or
        type(zmin) == float):
      self.zmin = zmin
    else:
      raise NumberCheckError('minimum in z-axis (zmin)')

  def setZpoints( self, zpoints = 5 ):
    """
    Method used to set the zpoints (accuracy in z-axis) parameter.
    """
    if( type(zpoints) == int or
        type(zpoints) == float ):
      self.zpoints = zpoints
    else:
      raise NumberCheckError('points in z-axis (zpoints)', type_=int)

  def setYpoints( self, ypoints = 100 ):
    """
    Method used to set the ypoints (accuracy in y-axis) parameter.
    """
    if( type(ypoints) == int or 
        type(ypoints) == float ):
        self.ypoints = ypoints
    else:
      raise NumberCheckError('points in y-axis (ypoints)', type_=int)

  def setN( self, N = 50 ):
    """
    Method used to set the N (number of terms in series developes) parameter.
    """
    if( type(N) == int or 
        type(N) == float ):
      self.N = N
    else:
      raise NumberCheckError('order of fourier series used (N)', type_=int)

  def setCs( self, cs = 0 ):
    """
    Method used to set the Cs (initial concentration condition for each slug) parameter.
    """
    if( type(cs) == int or 
        type(cs) == float ):
      if( 0 <= cs <= 1 ):
        self.cs = cs
      else:
        raise SetterIntervalError('Initial concentration (cs)', 0, 1)
    else:
      raise NumberCheckError('Initial concentration (cs)')
    
  def setCeq( self, ceq = 1 ):
    """
    Method used to set the Ceq (equilibrium concentrate between gas and liquid) parameter.
    """
    if( type( ceq ) == int or 
        type( ceq ) == float ):
      self.ceq = ceq
    else:
      raise NumberCheckError('order of fourier series used (N)', type_=int)
    self.ceq = ceq

  def setUs( self, Us=0.2 ):
    """
    Method used to set the Us (average fluids velocity) parameter.
    After Us is setted, this method calls the setter methods of Cab, DeltaBubble, Ub and Cas.
    """
    if( type(Us) == int or 
        type(Us) == float ):
      self.Us = Us
      self.setCas( )
      self.setDeltaBubble( )
      self.setUb( )
      self.setCab( )
    else:
      raise NumberCheckError('order of average velocity (Us)', type_=int)

  def setCab( self, Cab = None ):
    """
    Method used to set the capillarity number.
    The calculations were done defining Cab as next expresion: Cab = viscosity * Us / surface tension.
    """
    if( type( Cab ) == type(None) ):
      if( type(self.Ub) != type(None) ):
        self.Cab = self.visc * self.Ub/ self.surften
      else:
        print('Error. Trying to definde Cab before Ub.')
    else:
      self.Cab = Cab

  def setCas( self, Cas = None ):
    """
    Method used to set the capillarity number.
    The calculations were done defining Cas as next expresion: Cas = viscosity * Us / surface tension.
    """
    if( type( Cas ) == type(None) ):
      if( type(self.Us) != type(None) ):
        self.Cas = self.visc * self.Us/ self.surften
      else:
        print('Error. Trying to definde Cas before Us.')
    else:
      self.Cas = Cas

  def setDeltaBubble( self, delta_b=None ):
    """
    Method used to set the delta in bubble region.
    The calculations were done by experimental expression (doi: 10.1063/1.1289396).
    """
    if( type(delta_b) == type(None) ):
      if( type(self.Cas) != type(None) ):
        self.delta_b = 0.55 * self.Cas**(2/3) / ( 1 + 0.55*2.5* self.Cas**(2/3) ) * self.rc
      else:
        print('Error. Trying to calcule delta_b before Ca.')
    else:
      self.delta_b = delta_b

  def setDeltaSlug( self ):
    """
    Private Method. Method that calculates the delta slug with the variables already setted in the model.
    """
    self.delta_s = (1 - math.sqrt( 2 - self.Ub/self.Us )) * self.rc
  
  def setUb( self,  Ub = None ):
    """
    Method used to set the Ub (bubble velocity) parameter.
    The calculations were done by experimental expression ( ** pending of complete ** ).
    """
    if( type(Ub) == type(None) ):
      if( type(self.delta_b) != type(None) ):
        
        Rb_R = (self.rc - self.delta_b)/self.rc
        cte = (self.dens * self.g)/(8 * self.visc)
        qf = cte * self.rc**4 * (1 + 4* (Rb_R)**4 * ( 3/4 - math.log(Rb_R) - (Rb_R)**(-2) ) )
        self.Ub = (math.pi * self.rc**2 * self.Us + qf)/(math.pi * (self.rc - self.delta_b)**2)
      else:
        print('Error. Trying to calculate Ub before delta_b.')
    else:
      self.Ub = Ub

  def setLSL( self,  LSL=None ):
    """
    Method used to set the LSL (length of the liquid phase) parameter.
    """
    if( type( LSL ) == type( None ) ):
      print('Error. LSL need to have a value.')
    self.LSL = LSL

  def setGSL( self, GSL=None ):
    """
    Method used to set the GSL (length of the gas phase) parameter.
    """
    if( type( GSL ) == type( None ) ):
      print('Error. GSL need to have a value.')
    self.GSL = GSL
    #self.setZmax( self.GSL/self.rc - 2)

  def setRf( self ):
    """
    Method used to set the Rf (radius of forward cap of the bubble) parameter.
    """
    self.rf = self.rc / ( 1 + 1.286 * (3 * self.Cab)**(2/3) )

  def setRb( self ):
    """
    Method used to set the Rb (radius of backward cap of the bubble) parameter.
    """
    self.rb = self.rc / ( 1 - 0.464 * (3 * self.Cab)**(2/3) )

  def setVs( self, Vs=None ):
    """
    Method used to set the Vs (volume of the slug or liquid phase between bubbles) parameter.
    """
    A_f = 3/2 * (2*self.Us - self.Ub) - 2/3 * self.rf**2/self.rc**2 * self.Us
    B_f = 35/12 * self.rf**2/self.rc**2 * self.Us
    thF = math.asin( math.sqrt( A_f/B_f ) )

    A_b = 3/2 * (2 * self.Us - self.Ub) - 2/3 * self.rb**2/self.rc**2 * self.Us
    B_b = 35/12 * self.rb**2/self.rc**2 * self.Us
    thB = math.asin( math.sqrt( A_b/B_b ) )

    VcapF = math.pi/3 * ( (self.rc - self.delta_s)**3*math.tan(thF) - self.rf**3 * (math.sin(thF)**2 * math.cos(thF) + (1-math.cos(thF))**2*(2 - math.cos(thF) ) ) )
    VcapB = math.pi/3 * ( (self.rc - self.delta_s)**3*math.tan(thB) - self.rb**3 * (math.sin(thB)**2 * math.cos(thB) + (1-math.cos(thB))**2*(2 - math.cos(thB) ) ) )
    Vcilindro = math.pi*(self.rc - self.delta_s)**2 * ( self.LSL - ( self.rc - self.delta_s )*(math.tan(thF) + math.tan(thB) ) )
    self.Vs = Vcilindro + VcapF + VcapB

  def setVb( self, Vb=None ):
    """
    Method used to set the Vb (volume a single bubble) parameter.
    """
    V_forward = 2/3 * math.pi * self.rf**3
    V_backward = 2/3 * math.pi * self.rb**3
    V_midbubble = math.pi * (self.GSL - self.rf - self.rb) * (self.rc - self.delta_b)**2
    self.Vb = V_forward + V_backward + V_midbubble

##
# Setters
##