# -*- coding: utf-8 -*-
"""oModel.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1Mb0xTjQUJvINskLM4SBarzOgWN65BpP3
"""

# -----------------------------------------------------------
# Main object of the package.
#
# (C) 2021 Borja García, Valladolid, Spain
# Released under MIT License (MIT)
# email borjagarmarpy@gmail.com
# -----------------------------------------------------------

import math 
import numpy as np

import sys
sys.path.append('/content/drive/MyDrive/Burbuja/Difusion-Trabajando/Metodos')

# import mCalcAux as mc
from varsSetter import vars
from customErrors import NumberCheckError
from customErrors import SetterIntervalError

class TaylorBubbleModel( vars ):
  
  """
  Main object of the model.

  This model allows to simulate a taylor's tube, i.e., a tube containing taylor's bubble.
  The aim of the system is to facilitate the exchange of the gas from the bubbles to the 
  liquid phase. So the liquid maintains an equilibrium with the gas dissolved in it. 
  Physical variables' values have to be introduced using a python dictionary. Variables wich 
  can be setted are: temperature (T), difusivity (D), radius of the tube (rc), average fluid 
  velocity of the liquid-gas system (Us), initial concentration (cs), equilibrium concentration (ceq),
  minimum value of z (zmin), maximum value of z (zmax), resolution of z-axis (zpoints), 
  resolution of y-axis (ypoints), number of terms in developments (N), longitudinal lenght of gas phase
  between bubbles (GSL) and lenght of bubbles without the caps (LSL). Furthermore, a parameter 
  (idealgas) determine if the gas loss in the transfer is taken in account or despised ( see
  example 3 for further information ).

  Note: z-axis represents the longitudinal distance in the tube and y-distance represents the radial
        distance inside the tube.
     
  Before making predictions, model should be initialized and velocity of liquid in bubble
  and slug region should be setted, using methods: setVelocityBubble and setVelocitySlug ( default
  option can be runned and it set the profile we propose ).
  
  The main methods that provide results are:
    - ConcentrationProfile: returns the concentration profile at at various points of the region determined.
      Caution! To calculate the concentration profile at the slug region, it is necesary to solve 
      first the bubble region, since it is considered the first part in
      the simulation.
    - CellConcentrationProfile: returns the concentration profile in both regions, first bubble is 
      solved and then the slug region.
    - simuleColumn: returns the result of the simulation of the whole column, i.e., the average
      concentration of gas concentration in each slug.
    - kla_aprox: returns the kla of a complete cell, considering that the mixture of liquid-gas is 
      homogeneous in the slug.
    - kla_film: returns the contribution of the film between the bubble and the wall tube to the kla.
    - kla_film: returns the contribution of the film to the kla.

  """

  # Physical Variables
  delta_b = None
  delta_s = None
  g   = 9.80665
  cs  = None

  # coeficients of cosine developes
  az_b = None
  az_s = None
  
  # Cell Parameters
  LSL = None # slug length
  GSL = None # bubble length

  # Parameter:
  param_bubble_loss = False

  def updateVars( self, args=dict() ):
    """
    Private method. It updates calculated parameters before other calculations are done.
    """
    # print('Updating some variables...')
    vars.setCas( self )
    if('delta_b' in args.keys() ):
      vars.setDeltaBubble( args['delta_b'] )
    else:
      vars.setDeltaBubble( self )
    vars.setUb( self )
    vars.setCab( self )
    vars.setRf( self )
    vars.setRb( self )
    vars.setDeltaSlug( self )
    vars.setVs( self )
    vars.setVb( self )

  def __init__(self, args = dict() ):
    """
    Initialize the model. It reads a dictionary to save valors neede for
    the simulation.
      
    Parameters:
      - args: dictionary with variable valors for the simulation. 
    """

    # Physical Variables
    try:
      if( 'T' in args.keys() ):
        vars.setT( self, args['T'] )
      else:
        vars.setT( self )
      
      vars.setVisc( self )
      vars.setSurfTens( self )
      vars.setDens( self )
      
      if( 'D' in args.keys() ):
        vars.setD( self, args['D'] )
      else:
        vars.setD( self )
      if( 'm' in args.keys() ):
        vars.setM( self, args['m'] )
      else:
        vars.setM( self )
      if( 'rc' in args.keys() ):
        vars.setRC( self, args['rc'] )
      else:
        vars.setRC( self )
      if( 'Us' in args.keys() ):
        vars.setUs( self, args['Us'] )
      else:
        vars.setUs( self )
      if( 'cs' in args.keys() ):
        vars.setCs( self, args['cs'] )
      else:
        vars.setCs( self )
      if( 'ceq' in args.keys() ):
        vars.setCeq( self, args['ceq'] )
      else:
        vars.setCeq( self )

      # In Silico Parameters
      if( 'zmin' in args.keys() ):
        vars.setZmin( self, args['zmin'] )
      else:
        vars.setZmin( self )
      if( 'zmax' in args.keys() ):
        vars.setZmax( self, args['zmax'] )
      else:
        vars.setZmax( self )
      if( 'zpoints' in args.keys() ):
        vars.setZpoints( self, args['zpoints'] )
      else:
        vars.setZpoints( self )
      if( 'ypoints' in args.keys() ):
        vars.setYpoints( self, args['ypoints'] )
      else:
        vars.setYpoints( self )
      if( 'N' in args.keys() ):
        vars.setN( self, args['N'] )
      else:
        vars.setN( self )
      if( 'GSL' in args.keys() ):
        vars.setGSL( self, args['GSL'] )
      else:
        vars.setGSL( self, (self.zmax + 2) * self.rc )
      if( 'LSL' in args.keys() ):
        vars.setLSL( self, args['LSL'] )
      else:
        vars.setLSL( self, self.zmax * self.rc)

      if( 'idealgas' in args.keys() ):
        if( type(args['idealgas']) == type(True) ):
          self.idealgas = args['idealgas']
        else:
          print("  Error. idealgas parameter has not been setted. The value should be 'True' or 'False'.")
      else:
        self.idealgas = True

      self.setVelocityBubble( )
      self.setVelocitySlug( )      
      self.updateVars( args )

    except NumberCheckError as nce:
      print( nce )
    except SetterIntervalError as sie:
      print( sie )
    except:
      print("Unexpected error:", sys.exc_info()[0])

  def getDeltaBubble( self ):
    """
    Public Method. Method retrieves the delta of the bubble if it has been previously calculated.
    """
    if( self.delta_b == None ):
      raise ValueError('Error. First set the value of delta on bubble.\nUse method setDeltaBubble and mode theoretical or experimental.')
    else:
      return self.delta_b

  def getDeltaSlug( self ):
    """
    Public Method. Method retrieves the delta of the slug if it has been previously calculated.
    """
    if( self.delta_s == None ):
      raise ValueError('Error. First set the value of delta on slug.\nUse method setDeltaSlug and mode theoretical or experimental.')
    else:
      return self.delta_s

  def getCBubble( self ):
    """
    Public Method. Method retrieves the cosines developes coeficients of velocity in bubble's region if it has been previously calculated.
    """
    if( type(self.c_b) == type( None ) ):
      raise ValueError('Error. First set the value of cosine development of velocity on bubble.\nUse method setVeloc_b.')
    else:
      return self.c_b

  def getCSlug( self ):
    """
    Public Method. Method retrieves the coeficients of velocity in slug's region if it has been previously calculated.
    """
    if( type(self.c_s) == type( None ) ):
      raise ValueError('Error. First set the value of cosine development of velocity on slug.\nUse method setVeloc_s.')
    else:
      return self.c_s     
      
  def toString( self ):
    """
    Public Method. Method prints main parameters of the model.
    """
    # Physical Variables
    print('Parameters of the Simulation:')
    print('       T: ' + "{:.2e}".format(self.T) + 'ºC' )
    print('    visc: ' + "{:.2e}".format(self.visc) + 'ºC' )
    print(' surften: ' + "{:.2e}".format(self.surften) + 'ºC' )
    print('    dens: ' + "{:.2e}".format(self.dens) + 'ºC' )
    print('       D: ' + "{:.2e}".format(self.D) + 'ºC' )
    print('       m: ' + "{:.2e}".format(self.m) + '' )

    print()
    print('      rc: ' + "{:.2e}".format(self.rc) + 'm' )
    print('      Us: ' + "{:.2e}".format(self.Us) + 'm/s' )
    print('      Ub: ' + "{:.2e}".format(self.Ub) + 'm/s' )
    print('     Cas: ' + "{:.2e}".format(self.Cas) + '' )
    print('     Cab: ' + "{:.2e}".format(self.Cab) + '' )
    print(' delta_b: ' + "{:.4e}".format(self.delta_b) + 'm' )
    print(' delta_s: ' + "{:.4e}".format(self.delta_s) + 'm' )
    print('      cs: ' + "{:.2e}".format(self.cs) + '' )
    print('     GSL: ' + "{:.2e}".format(self.GSL) + 'm' )
    print('     LSL: ' + "{:.2e}".format(self.LSL) + 'm' )

    print()
    # In Silico Parameters
    print('    zmin: ' + "{:.2e}".format(self.zmin) + ' min value for z axe' )
    print('    zmax: ' + "{:.2e}".format(self.zmax) + ' max value for z axe' )
    print(' zpoints: ' + "{:.2e}".format(self.zpoints) + ' number of points for z-axe' )
    print(' ypoints: ' + "{:.2e}".format(self.ypoints) + ' number of points for y-axe' )
    print('       N: ' + "{:.2e}".format(self.N) + ' number of terms for the developments' )


##############
############ Cosine Develope of Velocity ( C ) Managment
##############

  def _defaultVelocityBubble( self, y ):
    """
    Private Method. Method utilized as function of liquid's velocity in bubble region if no other function is introduced in method: setVelocityBubble manually.
    """
##    return ( self.Ub + self.dens*self.g*self.rc**2/(4*self.visc) * (1 - ( (self.rc-y)/self.rc)**2 + 2*((self.rc-self.delta_b)/self.rc)**2 * np.log((self.rc-y)/self.rc)) )
    # Both expressions are very similar, but the chosen one was localized in bibliograpy. I dont know where I find the first one.
    return ( self.Ub + self.dens*self.g/self.visc * ( self.delta_b * y - y**2/2 ) )

  def _defaultVelocitySlug( self, y ):
    """
    Private Method. Method utilized as function of liquid's velocity in slug region if no other function is introduced in method: setVelocitySlug manually.
    """
    return ( self.Ub - 2*self.Us*(1 - ( self.rc - y )**2/self.rc**2) )

  def setVelocityBubble( self, fun=None, args=None ):
    """
    Public Method. Method that sets the cosine developes coeficients of a given function.
    
    Parameters:
      - fun: Function that represent the profile of the velocity of the liquid in bubble region. It is used to calculated a serie of cosine develope coefficients as variables for further calculus.
      - args: Function introduced in previous parameter (fun) can have a parameter to introduce some variables that appears in velocity. This parameter (args) would be introduced as parameter to fun if it is necesary.
    """
    if( type(fun)==type(None) ):
      self.c_b = self._CosineDevelope( self._defaultVelocityBubble, range_=[0, self.getDeltaBubble()], N=self.N, accurate=1000 )
    else:
      self.c_b = self._CosineDevelope( fun, args=args, range_=[0, self.getDeltaBubble()], N=self.N, accurate=1000 )

  def setVelocitySlug( self, fun=None, args=None ):
    """
    Public Method. Method that sets the cosine developes coeficients of a given function.
    
    Parameters:
      - fun: Function that represent the profile of the velocity of the liquid in bubble region. It is used to calculated a serie of cosine develope coefficients as variables for further calculus.
      - args: Function introduced in previous parameter (fun) can have a parameter to introduce some variables that appears in velocity. This parameter (args) would be introduced as parameter to fun if it is necesary.
    """
    if( type(fun)==type(None) ):
      self.c_s = self._CosineDevelope( self._defaultVelocitySlug, range_=[0, self.getDeltaBubble()], N=self.N, accurate=1000 )
    else:
      self.c_s = self._CosineDevelope( fun, args=args, range_=[0, self.getDeltaBubble()], N=self.N, accurate=1000 )

  def _CosineDevelope( self, func, args = None, range_=[0, 1], N=50, accurate=200 ):
    """
    Private Method. Method that calculates the cosine developed coefficients of a given function.
    
    Parameters:
      - func: Function used for calculations.
      - args: Function introduced in previous parameter (func) can have a parameter to introduce some variables that appears in velocity. This parameter (args) would be introduced as parameter to func if it is necesary.
      - range_: range in which the cosine develope is calculated.
      - N: order of terms in the serie.
      - accurate: number of points used between the range. Determine the accurate of the calculations.
    """
    from scipy import integrate
    if( range_[1] != 0 ):
      range_[1] = 2 * range_[1]
    if( range_[0] != 0 ):
      range_[0] = 2 * range_[0]
    
    x = np.linspace( range_[0], range_[1], accurate )
    interv = range_[1] - range_[0]
    
    if( type(args) == type(None) ):
      coefs = np.array([ 1/(interv) * integrate.simps( func(x), x )  ])
    else:
      coefs = np.array([ 1/(interv) * integrate.simps( func(x, args), x )  ])
    
    for n in range(1, N):
      if( type(args) == type(None) ):
        integ = integrate.simps( 2/interv * func(x)*np.cos( n*math.pi/(interv) * x ) , x)
      else:
        integ = integrate.simps( 2/interv * func(x, args)*np.cos( n*math.pi/(interv) * x ) , x)
      coefs = np.append( coefs, integ )
    return coefs

  def _setA0Bubble( self ):
    """
    Private Method. Method that fixes the initial condition for the bubble of concentration profile (in cosine develope coefficients).
    """
    res = np.array( [] )
    if( type( self.az_s ) != type( None ) ):
      for n in range( self.N ):
        Dc = self.ceq - self.cs
        an = 4/( (n+1) * math.pi) * Dc * math.sin( (n+1) * math.pi / 2 ) - self.az_s[ n-1 ]
        res = np.append( res, an )
    else:
      for n in range(1, self.N+1):
        Dc = self.ceq - self.cs
        an = 4/(n * math.pi) * Dc * math.sin( n * math.pi / 2 )
        res = np.append( res, an )
    return res

  def _setA0Slug( self ):
    """
    Private Method. Method that fixes the initial condition for the slug of concentration profile (in cosine develope coefficients).
    """    
    Dc = self.ceq - self.cs
    res = np.array([])
    if( type( self.az_b ) != type( None ) ):
      for n in range(self.N):
        an = 4/( (n+1) * math.pi) * Dc * math.sin( (n+1) * math.pi / 2 ) - self.az_b[ n-1 ]
        res = np.append( res, an )
      return res
    else:
      print('Error. First calcule the bubble region.')

###
# End Cosine Develope of Velocity ( C ) Managment
###

##############
############ Concentrations Profiles
##############

  def ConcentrationProfile( self, region = 'bubble' ):
    """
    Public Method. Method returns the profile concentration of gas in liquid in the zone specified.
    The method create a new layer of control between the calculus and what is returned to the user.
    
    Parameters:
     - region: region where the method is solved. The possible values are: 'bubble' or 'slug'.

    Return:
     - y: multi-dimensional arrays with points of the y-axis in the bubble region.
     - z: multi-dimensional arrays with points of the z-axis in the bubble region.
     - solution: multi-dimensional arrays with points of the concentration profile in the slug region.
    """   
    y, z, solution, *_ = self._ZoneCProfile( region=region )
    return y, z, solution

  def _ZoneCProfile( self, region = 'bubble' ):
    """
    Private Method. Method returns the profile concentration of gas in liquid in the zone specified.
    This method is called inside other public method to allow get the result of the simulations.
    
    Parameters:
      - region: region where the method is solved. The possible values are: 'bubble' or 'slug'.
    """   
    self.updateVars(args)
    
    # inicial and contour conditions
    if( region in 'bubble'):
      c = self.getCBubble()
      a0 = self._setA0Bubble()
      delta = self.getDeltaBubble()
    elif( region in 'slug' ):
      c = self.getCSlug()
      a0 = self._setA0Slug()
      delta = self.getDeltaSlug()
    else:
      print('Region not valid. Valid parameter zone values are "bubble" or "slug".') 
      return -1, -1, -1, -1, -1

    z = np.linspace( self.zmin,  self.zmax, self.zpoints )
    y = np.linspace(         0,      delta, self.ypoints )
    #########
    
    # Data structures
    lambd = np.zeros( self.N, dtype=np.float64 )
    for n  in range( len( lambd ) ):
      lambd[ n ] = ( (n+1) * math.pi ) / ( 2 * delta )

    B  = self.D / c[0] * lambd**2 
    A = np.zeros( [self.N, self.N] )   # Defining Matrix Solution...

    # Maths:
    A[ 0, 0] = a0[0]
    for n in range(len(A)-1):
      AT    = A.T
      Gamma = 1 / (self.D * ( lambd[:n+1]**2 + lambd[n+1]**2) )
      F = -np.dot( AT[n, :n+1], c[1:n+2][::-1] ) * Gamma
      A[:n+1, n+1] = F                   # iterative terminus
      A[ n+1, n+1] = a0[n+1] - np.sum(F) # initial condition terminus

    # Sum the contribution of each exponential and evaluating in z
    an_z = np.array([])
    for col in A.T:
      sol = 0
      for coef, B_i in zip( col, B ):
        sol += coef * np.exp( - B_i * z)
      an_z = np.append( an_z, sol )
    an_z = an_z.reshape(self.N, -1)

    # Sum the contribution of each cosine and evaluating in y
    solutions = np.array([])
    for ai, lambda_i in zip( an_z, lambd ):
      sol = ai.reshape(len(z), 1) * np.cos( lambda_i * y.reshape(1, len(y)) )
      solutions = np.append( solutions, sol )
    solutions = solutions.reshape( self.N, len(z), len(y))

    # final solution
    if( region in 'bubble'):
      solution = 1 - np.sum( solutions, axis = 0 )
      self.az_b = an_z.swapaxes(0, 1)[-1]
    elif( region in 'slug'):
      solution = self.cs + np.sum( solutions, axis = 0 )
      self.az_s = an_z.swapaxes(0, 1)[-1]
    else:
      print('Region not valid. Valid parameter zone values are "bubble" or "slug".')
      return -1, -1, -1, -1, -1
    
    return y, z, solution, A, an_z


  def CellConcentrationProfile( self ):
    """
    Public Method. Method returns the profile concentration of gas in liquid in the zone specified.
    The method create a new layer of control between the calculus and what is returned to the user.
    
    Parameters:
     - region: region where the method is solved. The possible values are: 'bubble' or 'slug'.

    Return:
     - ys_b: multi-dimensional arrays with points of the y-axis in the bubble region.
     - ys_s: multi-dimensional arrays with points of the y-axis in the slug region.
     - zs_b: multi-dimensional arrays with points of the z-axis in the bubble region.
     - zs_s: multi-dimensional arrays with points of the z-axis in the slug region.
     - solution_b: multi-dimensional arrays with points of the concentration profile in the bubble region.
     - solution_s: multi-dimensional arrays with points of the concentration profile in the slug region.
    """   
    y_b, y_s, z_b, z_s, solution_b, solution_s, A_b_, A_s_ = self._CellCProfile( )
    return y_b, y_s, z_b, z_s, solution_b, solution_s


  def _CellCProfile( self ):
    """
    Private Method. Method returns the profile concentration of gas in liquid in the whole cell.
    This method is called inside other public method to allow get the result of the simulations.
    """   
    y_b, z_b, solution_b, A_b, an_z_b = self._ZoneCProfile(region = 'bubble')
    y_s, z_s, solution_s, A_s, an_z_s = self._ZoneCProfile(region =   'slug')
    return y_b, y_s, z_b, z_s, solution_b, solution_s, A_b, A_s


###
# End Concentrations Profiles
###

##############
############ J - Gas Stream
##############

  def _J_cap( self, R ):
    """
    Private Method. Generic method to calcule both forward and backward gas stream value.

    Parameters:
    - R: radius of forward or backward cap to complete the parameters needed for calculations.

    Return:
    - J: value of gas stream for the particular forward or backward case.
    """
    A = 5/4 * (2*self.Us - self.Ub) + 2/3 * R**2/self.rc**2 * self.Us
    B = -35/12 * R**2/self.rc**2 * self.Us
    y_1 = (A-B) * (1-(1-A/B)**0.5) + (2/3*B-1/3*A) * (1-(1-A/B)**1.5) - 1/5 * B * (1-(1-A/B)**2.5)
    integral = 2 * np.sqrt( y_1 )  
    J = 2 * np.sqrt( math.pi * self.D * R**3 ) * ( self.ceq - self.cs ) * integral
    return J

  def J_forward_cap( self ):
    """
    Public Method. Method returns the value of J (gas stream) for forward cap of the bubble.

    Return:
    - J: value of gas stream for forward cap.
    """
    return self._J_cap( self.rf )

  def J_backward_cap( self ):  
    """
    Public Method. Method returns the value of J (gas stream) for backward cap of the bubble.

    Return:
    - J: value of gas stream for backward cap.
    """
    return self._J_cap( self.rb )

  def J_caps( self ):
    """
    Public Method. Method returns the value of J (gas stream) for both caps of the bubble.

    Return:
    - J: value of gas stream for both caps.
    """
    return ( self.J_backward_cap() + self.J_forward_cap() )

  def J_film(self):
    """
    Public Method. Method returns the value of J (gas stream) for the film in the bubble region.

    Return:
    - J: value of gas stream for the film in the bubble region.
    """ 
    J_film = -( model.ceq - model.cs ) * math.pi * (model.Us * model.rc**2 - model.Ub * (model.rc-model.delta_b)**2 )
    return J_film

  def J_total(self):
    """
    Public Method. Method returns the value of J (gas stream) for whole bubble to liquid transference.

    Return:
    - J: value of gas stream for the whole bubble to liquid transference.
    """ 
    return ( self.J_film() + self.J_caps() )

###
# End of J - Gas stream
###

##############
############ Kla
##############

  def calculateF( self, region = 'bubble' ): 
    _, _, _, A, an_z = self._ZoneCProfile( region=region )
    self.updateVars(args)
    sum_n = 0
    for n, col in enumerate(A.T):
      sum_n = sum_n + col/(n+1) * np.sin( (n+1)*math.pi/2 )

    sum_i = 0
    for i, ai in enumerate( sum_n ):
      lamb = (i+1)*math.pi/( 2 * self.delta_b )
      alfa = self.D/self.Ub * lamb**2
      sum_i = sum_i + ai * ( 1 - np.exp( - alfa * self.LSL ) )
    return sum_i * self.Ub * 4 * self.delta_b * ( self.rc - self.delta_b )

  def nextCs( self, F ):
    """
    Public Method. Method that calculates the concentration in the liquid after the gas absorbed from the 
    bubble to the cell. It will be the initial value of concentration for the next cell.

    Paramters:
     - F: quantity of gas transfered from the bubble to the liquid. It can be the caps, film or the whole quantity of gas.

    Returns:
     - nCs: outgoing concentration gas for the cell
    """
    Dc  = self.ceq - self.cs
    exp = - F * (self.LSL+self.GSL)/ (self.Vs * self.Ub)   
    nCs = self.ceq - Dc * math.e**exp
    return nCs

  def _kla( self, F ):
    """
    Public Method. Method that calculates the kla for a given F (gas transmitted from the bubble to the liquid).

    Paramters:
     - F: quantity of gas transfered from the bubble to the liquid. It can be the caps, film or the whole quantity of gas.

    Returns:
     - nCs: initial concentration 
    """
    nextCs = self.nextCs( F )
    kla = self.Us / (self.LSL + self.GSL) * math.log( (self.ceq - self.cs)/(self.ceq - nextCs) )
    return kla

  def kla_film( self ):
    """
    Public Method. Method that calculates and return the contribution of the film between the bubble and the tube to the kla.

    Returns:
      - kla: contribution of film to the kla
    """   
    self.updateVars(args)
    F = self.J_film() + math.pi * self.Ub * ( 2*self.rc*self.delta_b - self.delta_b**2)
    return self._kla( F )

  def kla_forward_cap( self ):
    """
    Public Method. Method that calculates and return the contribution of the upper cap of the bubble to the kla.

    Returns:
      - kla: contribution of film to the kla
    """   
    self.updateVars(args)
    F = self.J_forward_cap()
    return self._kla( F )

  def kla_backward_cap( self ):
    """
    Public Method. Method that calculates and return the contribution of the lower cap of the bubble to the kla.

    Returns:
      - kla: contribution of film to the kla
    """   
    self.updateVars(args)
    F = self.J_backward_cap()
    return self._kla( F )


  def kla_caps( self ):
    """
    Public Method. Method that calculates and return the contribution of the caps of the bubble to the kla.
    Returns:
      - kla: contribution of caps to the kla
    """  
    F = self.J_caps()
    return self._kla( F )

  def kla( self ):
    """
    Public Method. Method that calculates and return the value of the kla.
    Returns:
      - kla: value of the kla
    """  
    self.updateVars(args)
    F = self.J_total() + math.pi * self.Ub * ( 2*self.rc*self.delta_b - self.delta_b**2 )
    return self._kla( F )

###
# End Concentrations Profiles
###


##############
############ Complete Column/Tube Simulation
##############

  def _calc_gass_lost( self, A ):
    """
    Private Method. Method that calculates the gas loss between the bubble and the liquid in the slug.
    It is used in the simulation of the column when the loss of gas is not depreciated.
    """
    self.updateVars(args)
    sum_n = 0
    for n, col in enumerate(A.T):
      sum_n = sum_n + (col/( (n+1)*math.pi))**3 * math.sin(n*math.pi/2)
    
    sum_i = 0
    for i, ai in enumerate( sum_n ):
      lamb = (i+1)*math.pi/( 2 * self.delta_b )
      alfa = self.D/self.Ub * lamb**2
      sum_i = sum_i + ai * (1 - np.exp( -alfa * self.GSL ) )
    return self.delta_b * self.ceq * self.GSL - (self.ceq - self.cs) * self.Ub/self.D * 8 * self.delta_b**4 * sum_i

  def _calc_Fs( self, A ):
    """
    Private Method. Method that calculates a necesary part of the column simulation coming from the gas exchange.
    """
    self.updateVars(args)
    sum_n = 0
    for n, col in enumerate(A.T):
      sum_n = sum_n + col/(n+1) * np.sin( (n+1)*math.pi/2 )

    sum_i = 0
    for i, ai in enumerate( sum_n ):
      lamb = (i+1)*math.pi/( 2 * self.delta_b )
      alfa = self.D/self.Ub * lamb**2
      sum_i = sum_i + ai * ( 1 - np.exp( - alfa * self.LSL ) )
    return sum_i * self.Ub * 4 * self.delta_b * ( self.rc - self.delta_b )

  def simuleColumn( self, cs = 0.9, nCell = None ):
    """
    Public Method. Method that calculates and return the properly structures to analyze the solution. 

    Parameters:
      - cs: final concentration of the gas when escapes out of the tube.
      - nCell: number of cells taken in account in the simulation. If it is not specified, the simulation
        will take all necessary cells to make the initial concentration null.
    
    Return:
     - ys_b: multi-dimensional arrays with points of the y-axis in the bubble region.
     - ys_s: multi-dimensional arrays with points of the y-axis in the slug region.
     - zs_b: multi-dimensional arrays with points of the z-axis in the bubble region.
     - zs_s: multi-dimensional arrays with points of the z-axis in the slug region.
     - sols_b: multi-dimensional arrays with points of the concentration profile in the bubble region.
     - sols_s: multi-dimensional arrays with points of the concentration profile in the slug region.
     - css: array with value of average concentration in the slug for every cell. This is equivalent to the gas concentration of the liquid between cells.
     - ceqs: array with the maximum values of the concentration the system could reach. If the parameter of the model 'param_bubble_loss' is setted true
       the value of ceqs will drop for each cell, but if it is false this effect would be depreciated. This value is directly relactionated with the amount
       of gas inside the bubble of the cell.
    """
    ys_b = np.array([])
    ys_s = np.array([])
    zs_b = np.array([])
    zs_s = np.array([])
    sols_b = np.array([])
    sols_s = np.array([])
    ceqs = np.array([])
    css = np.array([])

    self.cs = cs
    self.updateVars(args)

    cell_length = self.LSL + self.GSL

    if( type(nCell)==type(None) ):
      while True:
        
        self.ci = self.cs
        y_b, y_s, z_b, z_s, solution_b, solution_s, A_b, A_s = self._CellCProfile()

        Fs = self._calc_Fs(A_s)
        self.cs = self.cs - cell_length/(self.Ub * self.Vs) * Fs

        if( self.param_bubble_loss ):
          self.ceq = self.ceq - self._calc_gass_lost(A_b)
        else:
          pass

        if( self.cs < 0 ):
          break

        ys_b = np.append(ys_b, y_b)
        ys_s = np.append(ys_s, y_s)
        zs_b = np.append(zs_b, z_b)
        zs_s = np.append(zs_s, z_s)
        sols_b = np.append(sols_b, solution_b)
        sols_s = np.append(sols_s, solution_s)
        ceqs   = np.append( ceqs, self.ceq )
        css    = np.append( css, self.cs )

    else:
      for nCell_ in range(nCell):
        self.ci = self.cs
        y_b, y_s, z_b, z_s, solution_b, solution_s, A_b, A_s = self._CellCProfile()

        Fs = self._calc_Fs(A_s)
        self.cs = self.cs - cell_length/(self.Ub * self.Vs) * Fs

        if( self.param_bubble_loss ):
          self.ceq = self.ceq - self._calc_gass_lost(A_b)
        else:
          pass

        ys_b = np.append(ys_b, y_b)
        ys_s = np.append(ys_s, y_s)
        zs_b = np.append(zs_b, z_b)
        zs_s = np.append(zs_s, z_s)
        sols_b = np.append(sols_b, solution_b)
        sols_s = np.append(sols_s, solution_s)
        ceqs   = np.append( ceqs, self.ceq )
        css    = np.append(  css,  self.cs )

    return ys_b, ys_s, zs_b, zs_s, sols_b, sols_s, css, ceqs

###
# End Concentrations Profiles
###

