# -*- coding: utf-8 -*-
"""mFits.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1e3_rb__OBs0GGpFkIav7G-FVjH2YMof9
"""

from sklearn.pipeline import make_pipeline
from sklearn.linear_model import LinearRegression
from sklearn.preprocessing import PolynomialFeatures

import numpy as np

def density( temperature, substance = 'water'):
  ## Metodo que devuelve la densidad del agua para una temperatura determinada
  density_model = make_pipeline(PolynomialFeatures(3), LinearRegression())
  # Densidad / temperatura
  # Datos sacados de: https://www.usgs.gov/special-topic/water-science-school/science/water-density?qt-science_center_objects=0#qt-science_center_objects
  x = np.array([ 0, 4, 4.4, 10, 15.6, 21, 26.7, 32.2, 37.8, 48.9, 60, 71.1, 82.2, 93.3, 100 ])
  y = np.array([ 0.99987, 1.00000, 0.99999, 0.99975, 0.99907, 0.99802, 0.99669, 0.99510, 0.99318, 0.98870, 0.98338, 0.97729, 0.97056, 0.96333, 0.95865 ])

  density_model.fit(x[:, np.newaxis], y)
  
  if( substance == 'water'):
    return density_model.predict(np.array([temperature]).reshape(1, -1))[0] * 1000 # multiplico para cambio de unidades
  else:
    print("Substance not recognized.")


def viscosity(temperature, substance = 'water'):
  ## Metodo que devuelve la viscosidad del agua para una temperatura determinada
  viscosity_model = make_pipeline(PolynomialFeatures(3), LinearRegression())
  # Datos de viscosidades
  # Datos sacados de : https://www.engineersedge.com/physics/water__density_viscosity_specific_weight_13146.htm
  x  = np.array([ 10, 20, 30, 40, 50, 60, 70, 80, 90, 100])
  mu = np.array([ 1.308, 1.002, 0.7978, 0.6531, 0.5471, 0.4658, 0.4044, 0.3550, 0.3150, 0.2822])

  viscosity_model.fit(x[:, np.newaxis], mu)

  if( substance == 'water'):
    return viscosity_model.predict(np.array([temperature]).reshape(1, -1))[0] * 1e-3 # multiplico para pasar a SI
  else:
    print("Substance not recognized.")


def surface_tension(temperature, substance = 'water'):
  ## Metodo que devuelve la tension superficial del agua para una temperatura determinada
  ts_model = make_pipeline(PolynomialFeatures(3), LinearRegression())
  # Datos tension superficial
  # Sacados de: https://www.engineeringtoolbox.com/water-surface-tension-d_597.html
  x = np.array([ 0, 5, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100, 150, 200])
  y = np.array([ 7.56, 7.49, 7.42, 7.28, 7.12, 6.96, 6.79, 6.62, 6.44, 6.26, 6.08, 5.89, 4.82, 3.76 ])

  ts_model.fit(x[:, np.newaxis], y)

  if( substance == 'water'):
    return ts_model.predict(np.array([temperature]).reshape(1, -1))[0] * 1e-2 # multiplico para pasar a SI
  else:
    print("Substance not recognized.")

D_default   = 2.2e-9 # m2/s - valor difusion oxigeno en agua
T_default   = 25     # ºC   - temperatura tipica
RC_default  = 5e-3   # m    - ancho tubería
Utp_default = 0.2    # m/s - flujo liquido
#Ub_default  = 0.2    # m/s - veloc burbuja

Zmin_default  = 0.01
Zmax_default  = 5
Zpaso_default = 5
Ypaso_default = 100
N_default = 10

def setD( args = dict() ):
  if( "D" in args.keys() and args["D"] != None ):
    return args["D"]
  else:
    return D_default

def setT( args = dict() ):
  if( "T" in args.keys() and args["T"] != None ):
    return args["T"]
  else:
    return T_default

def setRC( args = dict() ):
  if( "rc" in args.keys() and args["rc"] != None ):
    return args["rc"] 
  else:
    return  RC_default

def setVisc( args = dict() ):
  T = setT( args )
  if( "visc" in args.keys() and args["visc"] != None ):
    return args["visc"]
  else:
    return ( viscosity( temperature = T ) )

def setSurfTens( args = dict() ):
  T = setT( args )
  if( "surftens" in args.keys() and args["surftens"] != None ):
    return args["surftens"]
  else:
    return (surface_tension( temperature = T ) )

def setDens( args = dict() ):
  T = setT( args )
  if( "dens" in args.keys() and args["dens"] != None ):
    return args["dens"]
  else:
    return density( temperature = T )

def setZmax( args = dict() ):
  if( "zmax" in args.keys() and args["zmax"] != None ):
    return args["zmax"]
  elif( "GSL" in args.keys() ):
    return ( args['GSL'] - 2 * rc ) / rc
  else:
    return Zmax_default

def setZmin( args = dict() ):
  if( "zmin" in args.keys() and args["zmin"] != None ):
    return args["zmin"]
  else:
    return Zmin_default

def setZpaso( args = dict() ):
  if( "zpaso" in args.keys() and args["zpaso"] != None ):
    return args["zpaso"]
  else:
    return Zpaso_default

def setYpaso( args = dict() ):
  if( "ypaso" in args.keys() and args["ypaso"] != None ):
    return args["ypaso"]
  else:
    return Ypaso_default

def setN( args = dict() ):
  if( "N" in args.keys() and args["N"] != None ):
    return args["N"]
  else:
    return N_default

def setC0( args = dict() ):
  if( "c0" in args.keys() and args["c0"] != None ):
    return args["c0"]
  else:
    return 0

def setCs( args = dict() ):
  if( "cs" in args.keys() and args["cs"] != None ):
    return args["cs"]
  else:
    return 0

def setCEQ( args = dict() ):
  if( "ceq" in args.keys() and args["ceq"] != None ):
    return args["ceq"]
  else:
    return 0

## Calculo de Ca con cuando Utp no conocido
def calculoCa( args = dict() ):
  import numpy as np
  from scipy.optimize import fsolve
  Ub      = args['Ub']
  visc    = setVisc( args )
  surften = setSurfTens( args )

  beta = visc / surften * Ub
  def funcion( Ca ):
    return ( 1 - 0.81 * Ca**(0.33) ) * beta - Ca 
  return float( fsolve( funcion, 0.01) )

# Devuelve Ub, Utp y Ca a la vez, las 3 estan interconectadas
def setUb_Utp_Ca( args ):
  if( 'Utp' in args.keys() 
  and not 'Ub' in args.keys() and args["Utp"] != None ):
    Utp = args['Utp']
    visc = setVisc( args )
    surten = setSurfTens( args )
    Ca = visc * Utp/ surten
    Ub = Utp /( 1 - 0.61 * Ca **0.33)
  elif( 'Ub' in args.keys() and not 'Utp' in args.keys() and args["Ub"] != None ):
    Ub  = args['Ub']
    Ca  = calculoCa( args )
    Utp = Ub * ( 1 - 0.61 * Ca **0.33)
  else:
    Utp = Utp_default
    visc = setVisc( args )
    surten = setSurfTens( args )
    Ca = visc * Utp/ surten
    Ub = Utp /( 1 - 0.61 * Ca **0.33)
  return Ub, Utp, Ca

def setUb( args ):
  Ub, _, _ = setUb_Utp_Ca( args )
  return Ub

def setUtp( args ):
  _, Utp, _ = setUb_Utp_Ca( args )
  return Utp

def setCa( args ):
  _, _, Ca = setUb_Utp_Ca( args )
  return Ca

def setLSL( args ):
  if( "LSL" in args.keys() ):
    return args["LSL"]
  else:
    zmax = setZmax( args )
    rc = setRC( args )
    return zmax * rc

def setGSL( args ):
  if( "GSL" in args.keys() ):
    return args["GSL"]
  else:
    zmax = setZmax( args )
    rc = setRC( args )
    return zmax * rc + 2 * rc