# AUTOGENERATED! DO NOT EDIT! File to edit: DataPipelineNotebooks/1.ParseGFS.ipynb (unless otherwise specified).

__all__ = ['ParseGFS']

# Cell
import xarray as xr
import matplotlib.pyplot as plt
import pandas as pd
import salem
import numpy as np
import geopandas as gpd
import datetime
from joblib import Parallel, delayed
import os
import time

# Cell
class ParseGFS:
    """Class which provides the basic utilities and processing to transform a set of GFS hourly weather file
       to a set of filtered, aggregated and optionally interpolated netCDF files organized.
    """
    @staticmethod
    def season_to_snow_start_date(season):
        if season == '15-16':
            return '2015-11-01'
        elif season == '16-17':
            return '2016-11-01'
        elif season == '17-18':
            return '2017-11-01'
        elif season == '18-19':
            return '2018-11-01'
        elif season == '19-20':
            return '2019-11-01'

    def __init__(self, season, state, data_root, interpolate=1):
        """Initialize the class

        Keyword arguments:
        season: the season code (e.g., 15-16, 16-17) for the season you are processing
        state: the name of the state or country we are processing
        data_root: the root path of the data folders which contains the 1.RawWeatherData folder
        interpolate: the degree of interpolation (1x and 4x have been tested, 1x is default)
        """
        self.season = season
        self.snow_start_date = ParseGFS.season_to_snow_start_date(season)
        self.state = state
        self.interpolate = interpolate
        self.data_root = data_root
        self.state_path = None

        #make sure these are correct, but these generally don't need to change
        if state == 'Washington':
            self.state_path = 'Washington'
        else:
            self.state_path = 'ColoradoUtah' #the nc file which contains the weather data contains both utah and colorado


        #Path to USAvalancheRegions.geojson in this repo
        self.region_path = '../Data'
        #path to the gfs netcdf files for input
        self.dataset_path = data_root + '/1.RawWeatherData/gfs/' + season + '/' + self.state_path + '/'
        #output path for the result of the interpolation
        self.day_path = data_root + '2.GFSDaily' + str(interpolate) + 'xInterpolation/'+ season + '/'

        self.filtered_path = data_root + '3.GFSFiltered' + str(interpolate) + 'xInterpolation/'+ season + '/'

        #input file pattern as we'll read a full winter season in one pass
        self.file_pattern2 = '00.f0[0-2]*.nc'

        p = 181
        if season in ['15-16', '19-20']:
            p = 182 #leap years

        self.date_values_pd = pd.date_range(self.snow_start_date, periods=p, freq="D")

        print(self.dataset_path + ' Is Input Directory')
        print(self.day_path + ' Is output directory and input to filtering')
        print(self.filtered_path + ' Is output directory of filtering')

        #check dates end on April 30 which is the last day we support
        assert(self.date_values_pd[-1].month == 4)
        assert(self.date_values_pd[-1].day == 30)

        if not os.path.exists(self.day_path):
            os.makedirs(self.day_path)

        if not os.path.exists(self.filtered_path):
            os.makedirs(self.filtered_path)


    def resample(self, t):
        """
        Convert netcdf files which already pivot across levels
        from a full forcecast file which covers many days
        to one which only covers one day in the future
        also changes the data from hourly to daily min, avg, and max values

        Keyword arguments:
        t: the pandas datetime to process
        """

        print('On time: ' + str(t) + '\n')

        try:
            with xr.open_mfdataset(self.dataset_path + 'gfs.0p25.' + t + self.file_pattern2, combine='nested', concat_dim='time', parallel=False) as ds:

                #make sure we are just getting the first 24 hours
                pd_t = pd.to_datetime(t)
                ds = ds.where(ds.time.dt.day == pd_t.day, drop=True)

                min_name_dict = {}
                max_name_dict = {}
                avg_name_dict = {}
                #print("Calcualting Variables")
                for k in ds.data_vars.keys():
                    min_name_dict[k] = k + '_min'
                    max_name_dict[k] = k + '_max'
                    avg_name_dict[k] = k + '_avg'

                #print("Reampling")
                resampled_ds = ds.resample(time='1d')
                min_1day = resampled_ds.min().rename(min_name_dict)
                max_1day= resampled_ds.max().rename(max_name_dict)
                avg_1day = resampled_ds.mean().rename(avg_name_dict)

                #print("Merging")
                merged_ds = xr.merge([min_1day, max_1day, avg_1day])
                try:
                    file = self.day_path + self.state_path + '_' + t + '.nc'
                    try:
                        if os.path.exists(file):
                            os.remove(file)
                    except OSError as e:
                        #can likely ignore
                        print('had remove error ' + format(e))
                        time.sleep(1)
                    merged_ds.to_netcdf(file)
                    merged_ds.close()
                except Exception as err:
                    return self.day_path + self.state_path + '_' + t + '.nc' + ' -- ' + format(err)
                    merged_ds.close()
        except OSError as err:
            print('Missing files for time: ' + t)

    @staticmethod
    def check_resample(dates):
        """
        method to check if there are any file open issues with the newly output files

        Keyword arguments:
        date: pandas dates to check
        """
        dates_with_errors = []
        for t in dates.strftime('%Y%m%d'):
            try:
                with xr.open_dataset(day_path + statePath + '_' + t + '.nc') as file:
                    file.close()
                continue
            except:
                dates_with_errors.append(t)
        return dates_with_errors

    def resample_local(self, jobs=4):
        """
        Executes the resample process on the local machine.
        Process is IO bound so don't overallocate n_jobs
        All-Nan Slice and Divide warnings can be ignored

        Keyword arguments:
        jobs: number of parallel processs to use (default = 4)
        """
        results = Parallel(n_jobs=jobs, backend="multiprocessing")(map(delayed(self.resample), self.date_values_pd.strftime('%Y%m%d')))

        redo = None
        if not any(results):
            print('No Errors')
        else:
            print('Errors in some files')
            #get the dates for the files which had errors
            redo = [x.split('_')[1].split('.')[0] for x in results if x != None]

        if redo is not None:
            #a bit of a manual process to find and fix any errors which were introduced.
            #i'm not entirely sure why some of these errors are non-deterministic but
            #retrying them fixes them if there are no data corruption issues
            #for data corruption we just skip the data
            #sometimes the file is corrupt or locked so you need to make sure its deleted first

            #fix any errors
            redo2 = []
            for r in redo:
                b = self.resample(r)
                if b is not None:
                    redo2.append(b)

            #another pass to try and fix any file corruption issues
            redo = self.check_resample(date_values_pd)

            results = Parallel(n_jobs=1, backend="multiprocessing")(map(delayed(self.resample), redo))

        return results


    def interpolate_and_write(self, t):
        """
        interpolate and filter each day
        don't use dask for this, much faster to process the files in parallel

        Keyword arguments:
        t: the pandas datetime to process
        """
        #Read in all avy region shapes and metadata
        regions_df = gpd.read_file(self.region_path + '/USAvalancheRegions.geojson')
        #filter to just the ones where we have lables for training
        training_regions_df = regions_df[regions_df['is_training']==True].copy()

        #TODO: this needs to not rely on a code change to add a region
        if self.state == 'Washington':
            training_regions_df = training_regions_df[training_regions_df['center']=='Northwest Avalanche Center']
        elif self.state == 'Utah':
            training_regions_df = training_regions_df[training_regions_df['center']=='Utah Avalanche Center']
        elif self.state == 'Colorado':
            training_regions_df = training_regions_df[training_regions_df['center']=='Colorado Avalanche Information Center']

        training_regions_df.reset_index(drop=True, inplace=True)
        #open files which were previously resampled to a single day per file
        try:
            with xr.open_dataset(self.day_path + self.state_path + '_' + t + '.nc') as tmp_ds:
                print('On time: ' + str(t))

                new_lon = np.linspace(tmp_ds.longitude[0], tmp_ds.longitude[-1], tmp_ds.dims['longitude'] * self.interpolate)
                new_lat = np.linspace(tmp_ds.latitude[0], tmp_ds.latitude[-1], tmp_ds.dims['latitude'] * self.interpolate)
                interpolated_ds = tmp_ds.interp(latitude=new_lat, longitude=new_lon)

                subsets = []
                filenames = []
                errors = []
                redo_date = []
                date = tmp_ds.time.dt.strftime('%Y%m%d').values[0]
                for _, row in training_regions_df.iterrows():
                    #print("Calculating region: " + row['name'])
                    f = self.filtered_path + 'Region_' + row['name'] + '_' + date + '.nc'
                    try:
                        if self.interpolate == 1:
                            tmp_subset = interpolated_ds.salem.subset(geometry=row['geometry'])
                        else:
                            tmp_subset = interpolated_ds.salem.subset(geometry=row['geometry']).salem.roi(geometry=row['geometry'])
                    except ValueError:
                        errors.append('Value Error: Ensure the correct training regions have been provided')

                        continue

                    try:
                        tmp_subset.to_netcdf(f)
                    except Exception as err:
                        os.remove(f)
                        errors.append(f + ' -- ' + format(err))
                        redo_date.append(date)
                        continue
        except OSError as err:
            print('Missing files for time: ' + t + 'with error ' + format(err))
            return None

        return (errors, redo_date)

    def interpolate_and_write_local(self, jobs=6):
        """
        Executes the interpolate and write process on the local machine.
        Process is IO bound so don't overallocate n_jobs

        Keyword arguments:
        jobs: number of parallel processs to use (default = 6)
        """

        results = Parallel(n_jobs=6, backend="multiprocessing")(map(delayed(self.interpolate_and_write), self.date_values_pd.strftime('%Y%m%d')))

        all_none = True
        for x in results:
            if x[0] == [] and x[1] == []:
                continue
            else:
                all_none = False
                break

        if all_none:
            print('No Errors, go to ConvertToZarr')
        else:
            print('errors in some files, redo those by calling interpolate_and_write on those specific files')

        return results