#!/usr/bin/env python3

import json
from requests.compat import urlparse, urlunparse, urlencode
import asyncio
import aiohttp
import websockets
import ssl

from embypy import __version__


class WebSocket:
    '''Basic websocet that runs function when messages are recived

    Parameters
    ----------
    conn : embypy.utils.Connector
      connector object
    url : str
      uri of websocet server
    ssl_str : str
      path to the ssl certificate for confirmation
    '''
    def __init__(self, conn, url, ssl_str=None):
        self.on_message = []
        self.url	= url
        self.conn	= conn
        if type(ssl_str) == str:
            self.ssl = ssl.SSLContext(ssl.PROTOCOL_SSLv23)
            self.ssl.load_verify_locations(cafile=ssl_str)
        else:
            self.ssl = ssl_str

    def connect(self):
        '''Establish a connection'''
        # TODO - authenticate to emby
        self.loop.create_task(self.handler())

    async def handler(self):
        '''Handle loop, get and process messages'''
        self.ws = await websockets.connect(self.url, ssl=self.ssl)
        while self.ws:
            message = await self.ws.recv()
            for handle in self.on_message:
                if asyncio.iscoroutinefunction(handle):
                    await handle(self, message)
                else:
                    handle(self, message)

    async def send(message):
        if not self.ws:
            return False
        return await self.ws.send(message)

    def send_sync(message):
        return self.loop.run_until_complete(self.send(message))

    def close(self):
        '''close connection to socket'''
        self.ws.close()
        self.ws = None


class Connector:
    '''Class responsible for comunication with emby

    Parameters
    ----------
    url : str
      url to connect to
    address-remote : str, optional
      alt url to connect to, pulic facing (see notes)
    ssl : str, optional
      path to ssl certificate - for self signed certs
    userid : str, optional
      emby id of the user you wish to connect as
    api-key : str
      api key generated by emby, used for authentication
    token : str
      similar to api key, but is meant for user logins
    username : str, optional
      username for login (see notes)
    password : str, optional
      password for login (see notes)
    device_id : str
      device id as registered in emby
    timeout : int
      number of seconds to wait before timeout for a request
    tries : int
      number of times to try a request before throwing an error
    loop : asyncio.AbstractEventLoop
      if given all calls should be awaitable
    jellyfin : bool
      if this is a jellyfin (false = emby) server

    Notes
    -----
    This class/object should NOT be used (except internally).

    Tf a address-remote url is given, then that will be used for output,
    such as the `embypy.objects.EmbyObject.url` atribute.

    `url` will always be used when making requests - thus I recomend using
    the local address for `url` and the remote address
    for `address-remote`

    Jellyfin and emby have some url differences right now,
    so set jellyfin's url scheme to true/false
    [or None (default) for auto-detect]
    '''
    def __init__(self, url, **kargs):
        if ('api_key'  not in kargs or 'userid'   not in kargs) and \
           ('username' not in kargs or 'password' not in kargs):
            raise ValueError(
                'provide api key and userid or username/password'
            )

        urlremote	= kargs.get('address-remote')
        self.ssl	= kargs.get('ssl', True)
        self.userid	= kargs.get('userid')
        self.token	= kargs.get('token')
        self.api_key	= kargs.get('api_key', self.token)
        self.username	= kargs.get('username')
        self.password	= kargs.get('password')
        self.device_id	= kargs.get('device_id', 'EmbyPy')
        self.timeout	= kargs.get('timeout', 30)
        self.tries	= kargs.get('tries', 3)
        self.jellyfin	= kargs.get('jellyfin')
        self.loop	= kargs.get('loop', asyncio.get_event_loop())
        self.url	= urlparse(url)
        self.urlremote	= urlparse(urlremote) if urlremote else urlremote

        self.attempt_login = False

        if self.ssl and type(self.ssl) == str:
            self.ssl = ssl.SSLContext(ssl.PROTOCOL_SSLv23)
            self.ssl.load_verify_locations(cafile=self.ssl)

        conn = aiohttp.TCPConnector(ssl_context=self.ssl)

        self.session = aiohttp.ClientSession(
            headers={
                'Authorization': (
                    'MediaBrowser Client="{0}",Device="{0}",'
                    'DeviceId="{1}",'
                    'Version="{2}"'
                ).format('EmbyPy', self.device_id, __version__)
            },
            connector=conn
        )
        # connect to websocket is user wants to
        if 'ws' in kargs:
            self.ws = WebSocket(self, self.get_url(websocket=True), self.ssl)
        else:
            self.ws = None

    def __del__(self):
        try:
            Connector.sync_run(self.session.close())
        except:
            self.loop.run_until_complete(self.session.close())

    @staticmethod
    def sync_run(f):
        if asyncio.iscoroutinefunction(f):
            f = f()

        if asyncio.iscoroutine(f):
            return asyncio.get_event_loop().run_until_complete(f)
        elif callable(f):
            return f()
        else:
            return f

    def info_sync(self):
        return self.sync_run(self.info())

    @property
    def is_jellyfin_sync(self):
        return self.sync_run(self.is_jellyfin)

    def login_sync(self):
        return self.sync_run(self.login())

    def get_sync(self, *args, **kargs):
        return self.sync_run(self.get(*args, **kargs))

    def delete_sync(self, *args, **kargs):
        return self.sync_run(self.delete(*args, **kargs))

    def post_sync(self, *args, **kargs):
        return self.sync_run(self.post(*args, **kargs))

    def getJson_sync(self, *args, **kargs):
        return self.sync_run(self.getJson(*args, **kargs))

    async def login_if_needed(self):
        # authenticate to emby if password was given
        if self.password and self.username and not self.token:
            await self.login()

    async def info(self):
        return await self.getJson(
            '/system/info/public',
            remote=False
        )

    @property
    async def is_jellyfin(self):
        if self.jellyfin is None:
            info = await self.info()
            product = info.get('ProductName', '')
            self.jellyfin = 'jellyfin' in product.lower()
        return self.jellyfin

    async def login(self):
        if not self.username or self.attempt_login:
            return

        self.attempt_login = True
        try:
            data = await self.post(
                '/Users/AuthenticateByName',
                data={
                    'username': self.username,
                    'pw': self.password,
                },
                send_raw=True,
                format='json',
            )
            data = await data.json()

            self.token		= data.get('AccessToken', '')
            self.userid		= data.get('User', {}).get('Id')
            self.api_key	= self.token

            self.session._default_headers.update(
                {'X-MediaBrowser-Token': self.token}
            )
        finally:
            self.attempt_login = True

    def get_url(
        self, path='/', websocket=False, remote=True,
        attach_api_key=True, userId=None, pass_uid=False, **query
    ):
        '''construct a url for an emby request

        Parameters
        ----------
        path : str
          uri path(excluding domain and port) of get request for emby
        websocket : bool, optional
          if true, then `ws(s)` are used instead of `http(s)`
        remote : bool, optional
          if true, remote-address is used (default True)
        attach_api_key : bool, optional
          if true, apikey is added to the query (default True)
        userId : str, optional
          uid to use, if none, default is used
        pass_uid : bool, optional
          if true, uid is added to the query (default False)
        query : karg dict
          additional parameters to set (part of url after the `?`)

        Also See
        --------
          get :
          getJson :
          post :
          delete :

        Returns
        -------
        full url
        '''
        userId = userId or self.userid
        if attach_api_key and self.api_key:
            query.update({
                'api_key': self.api_key,
                'deviceId': self.device_id
            })
        if pass_uid:
            query['userId'] = userId

        if remote:
            url = self.urlremote or self.url
        else:
            url = self.url

        if websocket:
            scheme = url.scheme.replace('http', 'ws')
        else:
            scheme = url.scheme

        url = urlunparse(
            (scheme, url.netloc, path, '', '{params}', '')
        ).format(
            UserId	= userId,
            ApiKey	= self.api_key,
            DeviceId	= self.device_id,
            params	= urlencode(query)
        )

        return url[:-1] if url[-1] == '?' else url

    async def _process_resp(self, resp):
        if (not resp or resp.status == 401) and self.username:
            await self.login()
            await resp.close()
            return False
        return True

    def add_on_message(self, func):
        '''add function that handles websocket messages'''
        return self.ws.on_message.append(func)

    async def get(self, path, **query):
        '''return a get request

        Parameters
        ----------
        path : str
          same as get_url
        query : kargs dict
          additional info to pass to get_url

        See Also
        --------
          get_url :
          getJson :

        Returns
        -------
        requests.models.Response
          the response that was given
        '''
        url = self.get_url(path, **query)

        for i in range(self.tries+1):
            await self.login_if_needed()
            try:
                resp = await self.session.get(url, timeout=self.timeout)
                if await self._process_resp(resp):
                    return resp
                else:
                    continue
            except aiohttp.ClientConnectionError:
                if i >= self.tries:
                    raise aiohttp.ClientConnectionError(
                        'Emby server is probably down'
                    )

    async def delete(self, path, **query):
        '''send a delete request

        Parameters
        ----------
        path : str
          same as get_url
        query : kargs dict
          additional info to pass to get_url

        See Also
        --------
        get_url :

        Returns
        -------
        requests.models.Response
          the response that was given
        '''
        url = self.get_url(path, **query)

        for i in range(self.tries+1):
            await self.login_if_needed()
            try:
                resp = await self.session.delete(url, timeout=self.timeout)
                if await self._process_resp(resp):
                    return resp
                else:
                    continue
            except aiohttp.ClientConnectionError:
                if i >= self.tries:
                    raise aiohttp.ClientConnectionError(
                        'Emby server is probably down'
                    )

    async def post(self, path, data={}, send_raw=False, **params):
        '''sends post request

        Parameters
        ----------
        path : str
          same as get_url
        query : kargs dict
          additional info to pass to get_url

        See Also
        --------
          get_url :

        Returns
        -------
        requests.models.Response
          the response that was given
        '''
        url = self.get_url(path, **params)
        jstr = json.dumps(data)

        for i in range(self.tries+1):
            await self.login_if_needed()
            try:
                if send_raw:
                    resp = await self.session.post(
                        url,
                        data=data,
                        timeout=self.timeout
                    )
                else:
                    resp = await self.session.post(
                        url,
                        data=jstr,
                        timeout=self.timeout
                    )
                if await self._process_resp(resp):
                    return resp
                else:
                    continue
            except aiohttp.ClientConnectionError:
                if i >= self.tries:
                    raise aiohttp.ClientConnectionError(
                        'Emby server is probably down'
                    )

    async def getJson(self, path, **query):
        '''wrapper for get, parses response as json

        Parameters
        ----------
        path : str
          same as get_url
        query : kargs dict
          additional info to pass to get_url

        See Also
        --------
          get_url :
          get :

        Returns
        -------
        dict
          the response content as a dict
        '''
        for i in range(self.tries+1):
            try:
                return await (await self.get(path, **query)).json()
            except:
                if i >= self.tries:
                    raise
