import pytest

from viaconstructor.preview_plugins import gcode as gcodeparser


@pytest.mark.parametrize(
    (
        "gcode",
        "expected",
        "expected_state",
        "expected_minmax",
        "expected_size",
        "expected_len",
    ),
    [
        (
            [
                "(--------------------------------------------------)",
                "(Generator: viaConstructor)",
                "(Filename: tests/data/all.dxf)",
                "(--------------------------------------------------)",
                "",
                "UNKNOWN LINE",
                "G9999 (unsupported)",
                "",
                "G20 (Metric/inch)",
                "G21 (Metric/mm)",
                "G41 (Left Offsets)",
                "G42 (Right Offsets)",
                "G40 (No Offsets)",
                "G91 (Relative-Mode)",
                "G90 (Absolute-Mode)",
                "F100",
                "G64 P0.05",
                "M05 (Spindle off)",
                "M04 S100 (Spindle on / CCW)",
                "M06 T1",
                "M03 S10000 (Spindle on / CW)",
                "G04 P1 (pause in sec)",
                "G00 Z5.0",
                "G00 X0.0 Y0.0",
                "",
                "",
                "(--------------------------------------------------)",
                "(Level: 2)",
                "(Order: 0)",
                "(Object: 2.0)",
                "(Distance: 40.0mm)",
                "(Closed: False)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm none)",
                "(--------------------------------------------------)",
                "(- Depth: -9.0mm -)",
                "G00 Z5.0",
                "G00 X-20.0 Y30.0",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G01 X-20.0 Y30.0 Z-9.0",
                "G01 X20.0 Y30.0 Z-9.0",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 2)",
                "(Order: 1)",
                "(Object: 3.0)",
                "(Distance: 40.0mm)",
                "(Closed: False)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm none)",
                "(--------------------------------------------------)",
                "(- Depth: -9.0mm -)",
                "G00 Z5.0",
                "G00 X20.0 Y40.0",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G01 X20.0 Y40.0 Z-9.0",
                "G01 X-20.0 Y40.0 Z-9.0",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 2)",
                "(Order: 2)",
                "(Object: 6.0)",
                "(Distance: 48.983479342731485mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm outside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X42.0 Y20.0",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G03 X42.0 Y20.0 Z-9.0 I0.0 J0.0",
                "G03 X44.343146 Y14.343146 Z-9.0 I8.0 J0.0",
                "G03 X50.0 Y12.0 Z-9.0 I5.656854 J5.656854",
                "G03 X55.656854 Y14.343146 Z-9.0 I0.0 J8.0",
                "G03 X58.0 Y20.0 Z-9.0 I-5.656854 J5.656854",
                "G03 X55.656854 Y25.656854 Z-9.0 I-8.0 J-0.0",
                "G03 X50.0 Y28.0 Z-9.0 I-5.656854 J-5.656854",
                "G03 X44.343146 Y25.656854 Z-9.0 I0.0 J-8.0",
                "G03 X42.0 Y20.0 Z-9.0 I5.656854 J-5.656854",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 1)",
                "(Order: 3)",
                "(Object: 5.0)",
                "(Distance: 64.0mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm inside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X42.0 Y28.0",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G01 X42.0 Y28.0 Z-9.0",
                "G01 X58.0 Y28.0 Z-9.0",
                "G01 X58.0 Y12.0 Z-9.0",
                "G01 X42.0 Y12.0 Z-9.0",
                "G01 X42.0 Y28.0 Z-9.0",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 1)",
                "(Order: 4)",
                "(Object: 7.0)",
                "(Distance: 40.245869835682875mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm inside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X54.0 Y-2.0",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G01 X54.0 Y-2.0 Z-9.0",
                "G01 X66.0 Y-2.0 Z-9.0",
                "G02 X67.414214 Y-2.585786 Z-9.0 I0.0 J-2.0",
                "G02 X68.0 Y-4.0 Z-9.0 I-1.414214 J-1.414214",
                "G01 X68.0 Y-6.0 Z-9.0",
                "G02 X67.414214 Y-7.414214 Z-9.0 I-2.0 J-0.0",
                "G02 X66.0 Y-8.0 Z-9.0 I-1.414214 J1.414214",
                "G01 X54.0 Y-8.0 Z-9.0",
                "G02 X52.585786 Y-7.414214 Z-9.0 I0.0 J2.0",
                "G02 X52.0 Y-6.0 Z-9.0 I1.414214 J1.414214",
                "G01 X52.0 Y-4.0 Z-9.0",
                "G02 X52.585786 Y-2.585786 Z-9.0 I2.0 J0.0",
                "G02 X54.0 Y-2.0 Z-9.0 I1.414214 J-1.414214",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 1)",
                "(Order: 5)",
                "(Object: 0.0)",
                "(Distance: 48.9834793427315mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm inside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X8.0 Y-1.959434878635765e-15",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G02 X8.0 Y-0.0 Z-9.0 I0.0 J0.0",
                "G02 X5.656854 Y-5.656854 Z-9.0 I-8.0 J0.0",
                "G02 X-0.0 Y-8.0 Z-9.0 I-5.656854 J5.656854",
                "G02 X-5.656854 Y-5.656854 Z-9.0 I0.0 J8.0",
                "G02 X-8.0 Y0.0 Z-9.0 I5.656854 J5.656854",
                "G02 X-5.656854 Y5.656854 Z-9.0 I8.0 J-0.0",
                "G02 X0.0 Y8.0 Z-9.0 I5.656854 J-5.656854",
                "G02 X5.656854 Y5.656854 Z-9.0 I-0.0 J-8.0",
                "G02 X8.0 Y-0.0 Z-9.0 I-5.656854 J-5.656854",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 0)",
                "(Order: 6)",
                "(Object: 1.0)",
                "(Distance: 134.7045681925116mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm outside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X22.0 Y8.881784197001254e-17",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G03 X22.0 Y0.0 Z-9.0 I0.0 J0.0",
                "G03 X15.556349 Y15.556349 Z-9.0 I-22.0 J0.0",
                "G03 X0.0 Y22.0 Z-9.0 I-15.556349 J-15.556349",
                "G03 X-15.556349 Y15.556349 Z-9.0 I0.0 J-22.0",
                "G03 X-22.0 Y0.0 Z-9.0 I15.556349 J-15.556349",
                "G03 X-15.556349 Y-15.556349 Z-9.0 I22.0 J-0.0",
                "G03 X-0.0 Y-22.0 Z-9.0 I15.556349 J15.556349",
                "G03 X15.556349 Y-15.556349 Z-9.0 I0.0 J22.0",
                "G03 X22.0 Y0.0 Z-9.0 I-15.556349 J15.556349",
                "G00 Z5.0",
                "",
                "(--------------------------------------------------)",
                "(Level: 0)",
                "(Order: 7)",
                "(Object: 4.0)",
                "(Distance: 236.7632685100653mm)",
                "(Closed: True)",
                "(isPocket: False)",
                "(Depth: -9.0mm / -9.0mm)",
                "(Tool-Diameter: 4.0mm)",
                "(Tool-Offset: 2.0mm outside)",
                "(--------------------------------------------------)",
                "G00 Z5.0",
                "G00 X28.0 Y-20.0",
                "(- Depth: -9.0mm -)",
                "F100",
                "G01 Z-9.0",
                "F1000",
                "G03 X28.0 Y-20.0 Z-9.0 I0.0 J0.0",
                "G03 X30.0 Y-22.0 Z-9.0 I2.0 J0.0",
                "G01 X90.0 Y-22.0 Z-9.0",
                "G03 X92.0 Y-20.0 Z-9.0 I0.0 J2.0",
                "G01 X92.0 Y24.0 Z-9.0",
                "G03 X86.727922 Y36.727922 Z-9.0 I-18.0 J0.0",
                "G03 X74.0 Y42.0 Z-9.0 I-12.727922 J-12.727922",
                "G01 X46.0 Y42.0 Z-9.0",
                "G03 X33.272078 Y36.727922 Z-9.0 I0.0 J-18.0",
                "G03 X28.0 Y24.0 Z-9.0 I12.727922 J-12.727922",
                "G01 X28.0 Y-20.0 Z-9.0",
                "G00 Z5.0",
                "",
                "(- end -)",
                "G00 Z5.0",
                "M05 (Spindle off)",
                "G00 X0.0 Y0.0",
                "",
            ],
            [
                [{"X": 0, "Y": 0, "Z": 0}, {"Z": 5.0, "X": 0, "Y": 0}, "CW"],
                [{"Z": 5.0, "X": 0, "Y": 0}, {"X": 0.0, "Y": 0.0, "Z": 5.0}, "CW"],
                [{"X": 0.0, "Y": 0.0, "Z": 5.0}, {"Z": 5.0, "X": 0.0, "Y": 0.0}, "CW"],
                [
                    {"Z": 5.0, "X": 0.0, "Y": 0.0},
                    {"X": -20.0, "Y": 30.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": -20.0, "Y": 30.0, "Z": 5.0},
                    {"Z": -9.0, "X": -20.0, "Y": 30.0},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": -20.0, "Y": 30.0},
                    {"X": -20.0, "Y": 30.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -20.0, "Y": 30.0, "Z": -9.0},
                    {"X": 20.0, "Y": 30.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 20.0, "Y": 30.0, "Z": -9.0},
                    {"Z": 5.0, "X": 20.0, "Y": 30.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 20.0, "Y": 30.0},
                    {"Z": 5.0, "X": 20.0, "Y": 30.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 20.0, "Y": 30.0},
                    {"X": 20.0, "Y": 40.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 20.0, "Y": 40.0, "Z": 5.0},
                    {"Z": -9.0, "X": 20.0, "Y": 40.0},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": 20.0, "Y": 40.0},
                    {"X": 20.0, "Y": 40.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 20.0, "Y": 40.0, "Z": -9.0},
                    {"X": -20.0, "Y": 40.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -20.0, "Y": 40.0, "Z": -9.0},
                    {"Z": 5.0, "X": -20.0, "Y": 40.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": -20.0, "Y": 40.0},
                    {"Z": 5.0, "X": -20.0, "Y": 40.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": -20.0, "Y": 40.0},
                    {"X": 42.0, "Y": 20.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 20.0, "Z": 5.0},
                    {"Z": -9.0, "X": 42.0, "Y": 20.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 20.0, "Z": -9.0},
                    {"X": 42.0, "Y": 20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 20.0, "Z": -9.0},
                    {"X": 42.159467, "Y": 18.410645, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.159467, "Y": 18.410645, "Z": -9.0},
                    {"X": 42.631512, "Y": 16.884653, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.631512, "Y": 16.884653, "Z": -9.0},
                    {"X": 43.397315, "Y": 15.48286, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 43.397315, "Y": 15.48286, "Z": -9.0},
                    {"X": 44.343146, "Y": 14.343146, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 44.343146, "Y": 14.343146, "Z": -9.0},
                    {"X": 44.343146, "Y": 14.343146, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 44.343146, "Y": 14.343146, "Z": -9.0},
                    {"X": 45.57975, "Y": 13.332063, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 45.57975, "Y": 13.332063, "Z": -9.0},
                    {"X": 46.992575, "Y": 12.58681, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 46.992575, "Y": 12.58681, "Z": -9.0},
                    {"X": 48.525297, "Y": 12.137097, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 48.525297, "Y": 12.137097, "Z": -9.0},
                    {"X": 50.0, "Y": 12.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 50.0, "Y": 12.0, "Z": -9.0},
                    {"X": 50.0, "Y": 12.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 50.0, "Y": 12.0, "Z": -9.0},
                    {"X": 51.589355, "Y": 12.159467, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 51.589355, "Y": 12.159467, "Z": -9.0},
                    {"X": 53.115347, "Y": 12.631512, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.115347, "Y": 12.631512, "Z": -9.0},
                    {"X": 54.51714, "Y": 13.397315, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.51714, "Y": 13.397315, "Z": -9.0},
                    {"X": 55.656854, "Y": 14.343146, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 55.656854, "Y": 14.343146, "Z": -9.0},
                    {"X": 55.656854, "Y": 14.343146, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 55.656854, "Y": 14.343146, "Z": -9.0},
                    {"X": 56.667937, "Y": 15.57975, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 56.667937, "Y": 15.57975, "Z": -9.0},
                    {"X": 57.41319, "Y": 16.992575, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 57.41319, "Y": 16.992575, "Z": -9.0},
                    {"X": 57.862903, "Y": 18.525297, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 57.862903, "Y": 18.525297, "Z": -9.0},
                    {"X": 58.0, "Y": 20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 58.0, "Y": 20.0, "Z": -9.0},
                    {"X": 58.0, "Y": 20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 58.0, "Y": 20.0, "Z": -9.0},
                    {"X": 57.840533, "Y": 21.589355, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 57.840533, "Y": 21.589355, "Z": -9.0},
                    {"X": 57.368488, "Y": 23.115347, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 57.368488, "Y": 23.115347, "Z": -9.0},
                    {"X": 56.602685, "Y": 24.51714, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 56.602685, "Y": 24.51714, "Z": -9.0},
                    {"X": 55.656854, "Y": 25.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 55.656854, "Y": 25.656854, "Z": -9.0},
                    {"X": 55.656854, "Y": 25.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 55.656854, "Y": 25.656854, "Z": -9.0},
                    {"X": 54.42025, "Y": 26.667937, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.42025, "Y": 26.667937, "Z": -9.0},
                    {"X": 53.007425, "Y": 27.41319, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.007425, "Y": 27.41319, "Z": -9.0},
                    {"X": 51.474703, "Y": 27.862903, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 51.474703, "Y": 27.862903, "Z": -9.0},
                    {"X": 50.0, "Y": 28.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 50.0, "Y": 28.0, "Z": -9.0},
                    {"X": 50.0, "Y": 28.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 50.0, "Y": 28.0, "Z": -9.0},
                    {"X": 48.410645, "Y": 27.840533, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 48.410645, "Y": 27.840533, "Z": -9.0},
                    {"X": 46.884653, "Y": 27.368488, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 46.884653, "Y": 27.368488, "Z": -9.0},
                    {"X": 45.48286, "Y": 26.602685, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 45.48286, "Y": 26.602685, "Z": -9.0},
                    {"X": 44.343146, "Y": 25.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 44.343146, "Y": 25.656854, "Z": -9.0},
                    {"X": 44.343146, "Y": 25.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 44.343146, "Y": 25.656854, "Z": -9.0},
                    {"X": 43.332063, "Y": 24.42025, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 43.332063, "Y": 24.42025, "Z": -9.0},
                    {"X": 42.58681, "Y": 23.007425, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.58681, "Y": 23.007425, "Z": -9.0},
                    {"X": 42.137097, "Y": 21.474703, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.137097, "Y": 21.474703, "Z": -9.0},
                    {"X": 42.0, "Y": 20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 20.0, "Z": -9.0},
                    {"Z": 5.0, "X": 42.0, "Y": 20.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 42.0, "Y": 20.0},
                    {"Z": 5.0, "X": 42.0, "Y": 20.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 42.0, "Y": 20.0},
                    {"X": 42.0, "Y": 28.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 28.0, "Z": 5.0},
                    {"Z": -9.0, "X": 42.0, "Y": 28.0},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": 42.0, "Y": 28.0},
                    {"X": 42.0, "Y": 28.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 28.0, "Z": -9.0},
                    {"X": 58.0, "Y": 28.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 58.0, "Y": 28.0, "Z": -9.0},
                    {"X": 58.0, "Y": 12.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 58.0, "Y": 12.0, "Z": -9.0},
                    {"X": 42.0, "Y": 12.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 12.0, "Z": -9.0},
                    {"X": 42.0, "Y": 28.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.0, "Y": 28.0, "Z": -9.0},
                    {"Z": 5.0, "X": 42.0, "Y": 28.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 42.0, "Y": 28.0},
                    {"Z": 5.0, "X": 42.0, "Y": 28.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 42.0, "Y": 28.0},
                    {"X": 54.0, "Y": -2.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 54.0, "Y": -2.0, "Z": 5.0},
                    {"Z": -9.0, "X": 54.0, "Y": -2.0},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": 54.0, "Y": -2.0},
                    {"X": 54.0, "Y": -2.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.0, "Y": -2.0, "Z": -9.0},
                    {"X": 66.0, "Y": -2.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.0, "Y": -2.0, "Z": -9.0},
                    {"X": 66.0, "Y": -2.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.0, "Y": -2.0, "Z": -9.0},
                    {"X": 66.397339, "Y": -2.039867, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.397339, "Y": -2.039867, "Z": -9.0},
                    {"X": 66.778837, "Y": -2.157878, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.778837, "Y": -2.157878, "Z": -9.0},
                    {"X": 67.129285, "Y": -2.349329, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.129285, "Y": -2.349329, "Z": -9.0},
                    {"X": 67.414214, "Y": -2.585786, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.414214, "Y": -2.585786, "Z": -9.0},
                    {"X": 67.414214, "Y": -2.585786, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.414214, "Y": -2.585786, "Z": -9.0},
                    {"X": 67.666985, "Y": -2.894937, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.666985, "Y": -2.894937, "Z": -9.0},
                    {"X": 67.853298, "Y": -3.248144, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.853298, "Y": -3.248144, "Z": -9.0},
                    {"X": 67.965726, "Y": -3.631324, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.965726, "Y": -3.631324, "Z": -9.0},
                    {"X": 68.0, "Y": -4.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 68.0, "Y": -4.0, "Z": -9.0},
                    {"X": 68.0, "Y": -6.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 68.0, "Y": -6.0, "Z": -9.0},
                    {"X": 68.0, "Y": -6.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 68.0, "Y": -6.0, "Z": -9.0},
                    {"X": 67.960133, "Y": -6.397339, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.960133, "Y": -6.397339, "Z": -9.0},
                    {"X": 67.842122, "Y": -6.778837, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.842122, "Y": -6.778837, "Z": -9.0},
                    {"X": 67.650671, "Y": -7.129285, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.650671, "Y": -7.129285, "Z": -9.0},
                    {"X": 67.414214, "Y": -7.414214, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.414214, "Y": -7.414214, "Z": -9.0},
                    {"X": 67.414214, "Y": -7.414214, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.414214, "Y": -7.414214, "Z": -9.0},
                    {"X": 67.105063, "Y": -7.666985, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 67.105063, "Y": -7.666985, "Z": -9.0},
                    {"X": 66.751856, "Y": -7.853298, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.751856, "Y": -7.853298, "Z": -9.0},
                    {"X": 66.368676, "Y": -7.965726, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.368676, "Y": -7.965726, "Z": -9.0},
                    {"X": 66.0, "Y": -8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 66.0, "Y": -8.0, "Z": -9.0},
                    {"X": 54.0, "Y": -8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.0, "Y": -8.0, "Z": -9.0},
                    {"X": 54.0, "Y": -8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.0, "Y": -8.0, "Z": -9.0},
                    {"X": 53.602661, "Y": -7.960133, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.602661, "Y": -7.960133, "Z": -9.0},
                    {"X": 53.221163, "Y": -7.842122, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.221163, "Y": -7.842122, "Z": -9.0},
                    {"X": 52.870715, "Y": -7.650671, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.870715, "Y": -7.650671, "Z": -9.0},
                    {"X": 52.585786, "Y": -7.414214, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.585786, "Y": -7.414214, "Z": -9.0},
                    {"X": 52.585786, "Y": -7.414214, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.585786, "Y": -7.414214, "Z": -9.0},
                    {"X": 52.333015, "Y": -7.105063, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.333015, "Y": -7.105063, "Z": -9.0},
                    {"X": 52.146702, "Y": -6.751856, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.146702, "Y": -6.751856, "Z": -9.0},
                    {"X": 52.034274, "Y": -6.368676, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.034274, "Y": -6.368676, "Z": -9.0},
                    {"X": 52.0, "Y": -6.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.0, "Y": -6.0, "Z": -9.0},
                    {"X": 52.0, "Y": -4.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.0, "Y": -4.0, "Z": -9.0},
                    {"X": 52.0, "Y": -4.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.0, "Y": -4.0, "Z": -9.0},
                    {"X": 52.039867, "Y": -3.602661, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.039867, "Y": -3.602661, "Z": -9.0},
                    {"X": 52.157878, "Y": -3.221163, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.157878, "Y": -3.221163, "Z": -9.0},
                    {"X": 52.349329, "Y": -2.870715, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.349329, "Y": -2.870715, "Z": -9.0},
                    {"X": 52.585786, "Y": -2.585786, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.585786, "Y": -2.585786, "Z": -9.0},
                    {"X": 52.585786, "Y": -2.585786, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.585786, "Y": -2.585786, "Z": -9.0},
                    {"X": 52.894937, "Y": -2.333015, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 52.894937, "Y": -2.333015, "Z": -9.0},
                    {"X": 53.248144, "Y": -2.146702, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.248144, "Y": -2.146702, "Z": -9.0},
                    {"X": 53.631324, "Y": -2.034274, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 53.631324, "Y": -2.034274, "Z": -9.0},
                    {"X": 54.0, "Y": -2.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 54.0, "Y": -2.0, "Z": -9.0},
                    {"Z": 5.0, "X": 54.0, "Y": -2.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 54.0, "Y": -2.0},
                    {"Z": 5.0, "X": 54.0, "Y": -2.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 54.0, "Y": -2.0},
                    {"X": 8.0, "Y": -1.959435, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": 5.0},
                    {"Z": -9.0, "X": 8.0, "Y": -1.959435},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": 8.0, "Y": -1.959435},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -1.959435, "Z": -9.0},
                    {"X": 8.0, "Y": -0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -0.0, "Z": -9.0},
                    {"X": 8.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": 0.0, "Z": -9.0},
                    {"X": 7.840533, "Y": -1.589355, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 7.840533, "Y": -1.589355, "Z": -9.0},
                    {"X": 7.368488, "Y": -3.115347, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 7.368488, "Y": -3.115347, "Z": -9.0},
                    {"X": 6.602685, "Y": -4.51714, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 6.602685, "Y": -4.51714, "Z": -9.0},
                    {"X": 5.656854, "Y": -5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 5.656854, "Y": -5.656854, "Z": -9.0},
                    {"X": 5.656854, "Y": -5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 5.656854, "Y": -5.656854, "Z": -9.0},
                    {"X": 4.42025, "Y": -6.667937, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 4.42025, "Y": -6.667937, "Z": -9.0},
                    {"X": 3.007425, "Y": -7.41319, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 3.007425, "Y": -7.41319, "Z": -9.0},
                    {"X": 1.474703, "Y": -7.862903, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 1.474703, "Y": -7.862903, "Z": -9.0},
                    {"X": -0.0, "Y": -8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -0.0, "Y": -8.0, "Z": -9.0},
                    {"X": 0.0, "Y": -8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": -8.0, "Z": -9.0},
                    {"X": -1.589355, "Y": -7.840533, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -1.589355, "Y": -7.840533, "Z": -9.0},
                    {"X": -3.115347, "Y": -7.368488, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -3.115347, "Y": -7.368488, "Z": -9.0},
                    {"X": -4.51714, "Y": -6.602685, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -4.51714, "Y": -6.602685, "Z": -9.0},
                    {"X": -5.656854, "Y": -5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -5.656854, "Y": -5.656854, "Z": -9.0},
                    {"X": -5.656854, "Y": -5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -5.656854, "Y": -5.656854, "Z": -9.0},
                    {"X": -6.667937, "Y": -4.42025, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -6.667937, "Y": -4.42025, "Z": -9.0},
                    {"X": -7.41319, "Y": -3.007425, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -7.41319, "Y": -3.007425, "Z": -9.0},
                    {"X": -7.862903, "Y": -1.474703, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -7.862903, "Y": -1.474703, "Z": -9.0},
                    {"X": -8.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -8.0, "Y": 0.0, "Z": -9.0},
                    {"X": -8.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -8.0, "Y": 0.0, "Z": -9.0},
                    {"X": -7.840533, "Y": 1.589355, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -7.840533, "Y": 1.589355, "Z": -9.0},
                    {"X": -7.368488, "Y": 3.115347, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -7.368488, "Y": 3.115347, "Z": -9.0},
                    {"X": -6.602685, "Y": 4.51714, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -6.602685, "Y": 4.51714, "Z": -9.0},
                    {"X": -5.656854, "Y": 5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -5.656854, "Y": 5.656854, "Z": -9.0},
                    {"X": -5.656854, "Y": 5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -5.656854, "Y": 5.656854, "Z": -9.0},
                    {"X": -4.42025, "Y": 6.667937, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -4.42025, "Y": 6.667937, "Z": -9.0},
                    {"X": -3.007425, "Y": 7.41319, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -3.007425, "Y": 7.41319, "Z": -9.0},
                    {"X": -1.474703, "Y": 7.862903, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -1.474703, "Y": 7.862903, "Z": -9.0},
                    {"X": 0.0, "Y": 8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": 8.0, "Z": -9.0},
                    {"X": 0.0, "Y": 8.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": 8.0, "Z": -9.0},
                    {"X": 1.589355, "Y": 7.840533, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 1.589355, "Y": 7.840533, "Z": -9.0},
                    {"X": 3.115347, "Y": 7.368488, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 3.115347, "Y": 7.368488, "Z": -9.0},
                    {"X": 4.51714, "Y": 6.602685, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 4.51714, "Y": 6.602685, "Z": -9.0},
                    {"X": 5.656854, "Y": 5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 5.656854, "Y": 5.656854, "Z": -9.0},
                    {"X": 5.656854, "Y": 5.656854, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 5.656854, "Y": 5.656854, "Z": -9.0},
                    {"X": 6.667937, "Y": 4.42025, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 6.667937, "Y": 4.42025, "Z": -9.0},
                    {"X": 7.41319, "Y": 3.007425, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 7.41319, "Y": 3.007425, "Z": -9.0},
                    {"X": 7.862903, "Y": 1.474703, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 7.862903, "Y": 1.474703, "Z": -9.0},
                    {"X": 8.0, "Y": -0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.0, "Y": -0.0, "Z": -9.0},
                    {"Z": 5.0, "X": 8.0, "Y": -0.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 8.0, "Y": -0.0},
                    {"Z": 5.0, "X": 8.0, "Y": -0.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 8.0, "Y": -0.0},
                    {"X": 22.0, "Y": 8.881784, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": 5.0},
                    {"Z": -9.0, "X": 22.0, "Y": 8.881784},
                    "CW",
                ],
                [
                    {"Z": -9.0, "X": 22.0, "Y": 8.881784},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 8.881784, "Z": -9.0},
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    {"X": 21.561465, "Y": 4.370725, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 21.561465, "Y": 4.370725, "Z": -9.0},
                    {"X": 20.263342, "Y": 8.567204, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 20.263342, "Y": 8.567204, "Z": -9.0},
                    {"X": 18.157384, "Y": 12.422134, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 18.157384, "Y": 12.422134, "Z": -9.0},
                    {"X": 15.556349, "Y": 15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 15.556349, "Y": 15.556349, "Z": -9.0},
                    {"X": 15.556349, "Y": 15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 15.556349, "Y": 15.556349, "Z": -9.0},
                    {"X": 12.155688, "Y": 18.336827, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 12.155688, "Y": 18.336827, "Z": -9.0},
                    {"X": 8.270419, "Y": 20.386274, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.270419, "Y": 20.386274, "Z": -9.0},
                    {"X": 4.055433, "Y": 21.622984, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 4.055433, "Y": 21.622984, "Z": -9.0},
                    {"X": 0.0, "Y": 22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": 22.0, "Z": -9.0},
                    {"X": 0.0, "Y": 22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": 22.0, "Z": -9.0},
                    {"X": -4.370725, "Y": 21.561465, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -4.370725, "Y": 21.561465, "Z": -9.0},
                    {"X": -8.567204, "Y": 20.263342, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -8.567204, "Y": 20.263342, "Z": -9.0},
                    {"X": -12.422134, "Y": 18.157384, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -12.422134, "Y": 18.157384, "Z": -9.0},
                    {"X": -15.556349, "Y": 15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -15.556349, "Y": 15.556349, "Z": -9.0},
                    {"X": -15.556349, "Y": 15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -15.556349, "Y": 15.556349, "Z": -9.0},
                    {"X": -18.336827, "Y": 12.155688, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -18.336827, "Y": 12.155688, "Z": -9.0},
                    {"X": -20.386274, "Y": 8.270419, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -20.386274, "Y": 8.270419, "Z": -9.0},
                    {"X": -21.622984, "Y": 4.055433, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -21.622984, "Y": 4.055433, "Z": -9.0},
                    {"X": -22.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -22.0, "Y": 0.0, "Z": -9.0},
                    {"X": -22.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -22.0, "Y": 0.0, "Z": -9.0},
                    {"X": -21.561465, "Y": -4.370725, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -21.561465, "Y": -4.370725, "Z": -9.0},
                    {"X": -20.263342, "Y": -8.567204, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -20.263342, "Y": -8.567204, "Z": -9.0},
                    {"X": -18.157384, "Y": -12.422134, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -18.157384, "Y": -12.422134, "Z": -9.0},
                    {"X": -15.556349, "Y": -15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -15.556349, "Y": -15.556349, "Z": -9.0},
                    {"X": -15.556349, "Y": -15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -15.556349, "Y": -15.556349, "Z": -9.0},
                    {"X": -12.155688, "Y": -18.336827, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -12.155688, "Y": -18.336827, "Z": -9.0},
                    {"X": -8.270419, "Y": -20.386274, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -8.270419, "Y": -20.386274, "Z": -9.0},
                    {"X": -4.055433, "Y": -21.622984, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -4.055433, "Y": -21.622984, "Z": -9.0},
                    {"X": -0.0, "Y": -22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": -0.0, "Y": -22.0, "Z": -9.0},
                    {"X": 0.0, "Y": -22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 0.0, "Y": -22.0, "Z": -9.0},
                    {"X": 4.370725, "Y": -21.561465, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 4.370725, "Y": -21.561465, "Z": -9.0},
                    {"X": 8.567204, "Y": -20.263342, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 8.567204, "Y": -20.263342, "Z": -9.0},
                    {"X": 12.422134, "Y": -18.157384, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 12.422134, "Y": -18.157384, "Z": -9.0},
                    {"X": 15.556349, "Y": -15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 15.556349, "Y": -15.556349, "Z": -9.0},
                    {"X": 15.556349, "Y": -15.556349, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 15.556349, "Y": -15.556349, "Z": -9.0},
                    {"X": 18.336827, "Y": -12.155688, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 18.336827, "Y": -12.155688, "Z": -9.0},
                    {"X": 20.386274, "Y": -8.270419, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 20.386274, "Y": -8.270419, "Z": -9.0},
                    {"X": 21.622984, "Y": -4.055433, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 21.622984, "Y": -4.055433, "Z": -9.0},
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 22.0, "Y": 0.0, "Z": -9.0},
                    {"Z": 5.0, "X": 22.0, "Y": 0.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 22.0, "Y": 0.0},
                    {"Z": 5.0, "X": 22.0, "Y": 0.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 22.0, "Y": 0.0},
                    {"X": 28.0, "Y": -20.0, "Z": 5.0},
                    "CW",
                ],
                [
                    {"X": 28.0, "Y": -20.0, "Z": 5.0},
                    {"Z": -9.0, "X": 28.0, "Y": -20.0},
                    "CW",
                ],
                [
                    {"X": 28.0, "Y": -20.0, "Z": -9.0},
                    {"X": 28.0, "Y": -20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.0, "Y": -20.0, "Z": -9.0},
                    {"X": 28.039867, "Y": -20.397339, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.039867, "Y": -20.397339, "Z": -9.0},
                    {"X": 28.157878, "Y": -20.778837, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.157878, "Y": -20.778837, "Z": -9.0},
                    {"X": 28.349329, "Y": -21.129285, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.349329, "Y": -21.129285, "Z": -9.0},
                    {"X": 28.606587, "Y": -21.434712, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.606587, "Y": -21.434712, "Z": -9.0},
                    {"X": 28.919395, "Y": -21.682942, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.919395, "Y": -21.682942, "Z": -9.0},
                    {"X": 29.275284, "Y": -21.864078, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 29.275284, "Y": -21.864078, "Z": -9.0},
                    {"X": 29.660066, "Y": -21.970899, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 29.660066, "Y": -21.970899, "Z": -9.0},
                    {"X": 30.0, "Y": -22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 30.0, "Y": -22.0, "Z": -9.0},
                    {"X": 90.0, "Y": -22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 90.0, "Y": -22.0, "Z": -9.0},
                    {"X": 90.0, "Y": -22.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 90.0, "Y": -22.0, "Z": -9.0},
                    {"X": 90.397339, "Y": -21.960133, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 90.397339, "Y": -21.960133, "Z": -9.0},
                    {"X": 90.778837, "Y": -21.842122, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 90.778837, "Y": -21.842122, "Z": -9.0},
                    {"X": 91.129285, "Y": -21.650671, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.129285, "Y": -21.650671, "Z": -9.0},
                    {"X": 91.434712, "Y": -21.393413, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.434712, "Y": -21.393413, "Z": -9.0},
                    {"X": 91.682942, "Y": -21.080605, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.682942, "Y": -21.080605, "Z": -9.0},
                    {"X": 91.864078, "Y": -20.724716, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.864078, "Y": -20.724716, "Z": -9.0},
                    {"X": 91.970899, "Y": -20.339934, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.970899, "Y": -20.339934, "Z": -9.0},
                    {"X": 92.0, "Y": -20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 92.0, "Y": -20.0, "Z": -9.0},
                    {"X": 92.0, "Y": 24.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 92.0, "Y": 24.0, "Z": -9.0},
                    {"X": 92.0, "Y": 24.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 92.0, "Y": 24.0, "Z": -9.0},
                    {"X": 91.641198, "Y": 27.576048, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 91.641198, "Y": 27.576048, "Z": -9.0},
                    {"X": 90.579098, "Y": 31.00953, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 90.579098, "Y": 31.00953, "Z": -9.0},
                    {"X": 88.856041, "Y": 34.163565, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 88.856041, "Y": 34.163565, "Z": -9.0},
                    {"X": 86.727922, "Y": 36.727922, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 86.727922, "Y": 36.727922, "Z": -9.0},
                    {"X": 86.727922, "Y": 36.727922, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 86.727922, "Y": 36.727922, "Z": -9.0},
                    {"X": 83.945563, "Y": 39.002859, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 83.945563, "Y": 39.002859, "Z": -9.0},
                    {"X": 80.766706, "Y": 40.679679, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 80.766706, "Y": 40.679679, "Z": -9.0},
                    {"X": 77.318082, "Y": 41.691533, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 77.318082, "Y": 41.691533, "Z": -9.0},
                    {"X": 74.0, "Y": 42.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 74.0, "Y": 42.0, "Z": -9.0},
                    {"X": 46.0, "Y": 42.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 46.0, "Y": 42.0, "Z": -9.0},
                    {"X": 46.0, "Y": 42.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 46.0, "Y": 42.0, "Z": -9.0},
                    {"X": 42.423952, "Y": 41.641198, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 42.423952, "Y": 41.641198, "Z": -9.0},
                    {"X": 38.99047, "Y": 40.579098, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 38.99047, "Y": 40.579098, "Z": -9.0},
                    {"X": 35.836435, "Y": 38.856041, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 35.836435, "Y": 38.856041, "Z": -9.0},
                    {"X": 33.272078, "Y": 36.727922, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 33.272078, "Y": 36.727922, "Z": -9.0},
                    {"X": 33.272078, "Y": 36.727922, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 33.272078, "Y": 36.727922, "Z": -9.0},
                    {"X": 30.997141, "Y": 33.945563, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 30.997141, "Y": 33.945563, "Z": -9.0},
                    {"X": 29.320321, "Y": 30.766706, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 29.320321, "Y": 30.766706, "Z": -9.0},
                    {"X": 28.308467, "Y": 27.318082, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.308467, "Y": 27.318082, "Z": -9.0},
                    {"X": 28.0, "Y": 24.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.0, "Y": 24.0, "Z": -9.0},
                    {"X": 28.0, "Y": -20.0, "Z": -9.0},
                    "CW",
                ],
                [
                    {"X": 28.0, "Y": -20.0, "Z": -9.0},
                    {"Z": 5.0, "X": 28.0, "Y": -20.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 28.0, "Y": -20.0},
                    {"Z": 5.0, "X": 28.0, "Y": -20.0},
                    "CW",
                ],
                [
                    {"Z": 5.0, "X": 28.0, "Y": -20.0},
                    {"X": 0.0, "Y": 0.0, "Z": 5.0},
                    "OFF",
                ],
            ],
            {
                "move_mode": 0,
                "offsets": "OFF",
                "metric": "MM",
                "absolute": True,
                "feedrate": 1000.0,
                "tool": 1,
                "scale": 1.0,
                "spindle": {"dir": "OFF", "rpm": 10000.0},
                "position": {"X": 0.0, "Y": 0.0, "Z": 5.0},
            },
            [-22.0, -22.0, -9.0, 92.0, 42.0, 5.0],
            [114.0, 64.0, 14.0],
            302,
        ),
    ],
)
def test_GcodeParser_get_path(
    gcode, expected, expected_state, expected_minmax, expected_size, expected_len
):
    drawed_lines = []

    def draw_line(p_1, p_2, options):
        drawed_lines.append((p_1, p_2))

    parser = gcodeparser.GcodeParser(gcode)
    result = parser.get_path(rounding=True)
    state = parser.get_state()
    parser.draw(draw_line)
    print(result)
    assert result == expected
    assert state == expected_state
    assert parser.get_minmax() == expected_minmax
    assert parser.get_size() == expected_size
    assert len(drawed_lines) == expected_len
