"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.code_snippet_extension = void 0;
require("../style/index.css");
const ui_components_1 = require("@elyra/ui-components");
const application_1 = require("@jupyterlab/application");
const apputils_1 = require("@jupyterlab/apputils");
const codeeditor_1 = require("@jupyterlab/codeeditor");
const docregistry_1 = require("@jupyterlab/docregistry");
const fileeditor_1 = require("@jupyterlab/fileeditor");
const markdownviewer_1 = require("@jupyterlab/markdownviewer");
const notebook_1 = require("@jupyterlab/notebook");
const CodeSnippetService_1 = require("./CodeSnippetService");
const CodeSnippetWidget_1 = require("./CodeSnippetWidget");
const CODE_SNIPPET_EXTENSION_ID = 'elyra-code-snippet-extension';
const commandIDs = {
    saveAsSnippet: 'codesnippet:save-as-snippet'
};
/**
 * Initialization data for the code-snippet extension.
 */
exports.code_snippet_extension = {
    id: CODE_SNIPPET_EXTENSION_ID,
    autoStart: true,
    requires: [apputils_1.ICommandPalette, application_1.ILayoutRestorer, codeeditor_1.IEditorServices],
    optional: [apputils_1.IThemeManager],
    activate: (app, palette, restorer, editorServices, themeManager) => {
        console.log('Elyra - code-snippet extension is activated!');
        const getCurrentWidget = () => {
            return app.shell.currentWidget;
        };
        const codeSnippetWidget = new CodeSnippetWidget_1.CodeSnippetWidget({
            app,
            themeManager,
            display_name: 'Code Snippets',
            schemaspace: CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE,
            schema: CodeSnippetService_1.CODE_SNIPPET_SCHEMA,
            icon: ui_components_1.codeSnippetIcon,
            getCurrentWidget,
            editorServices
        });
        const codeSnippetWidgetId = `elyra-metadata:${CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE}`;
        codeSnippetWidget.id = codeSnippetWidgetId;
        codeSnippetWidget.title.icon = ui_components_1.codeSnippetIcon;
        codeSnippetWidget.title.caption = 'Code Snippets';
        restorer.add(codeSnippetWidget, codeSnippetWidgetId);
        // Rank has been chosen somewhat arbitrarily to give priority to the running
        // sessions widget in the sidebar.
        app.shell.add(codeSnippetWidget, 'left', { rank: 900 });
        app.commands.addCommand(commandIDs.saveAsSnippet, {
            label: 'Save As Code Snippet',
            isEnabled: () => {
                var _a, _b;
                const currentWidget = app.shell.currentWidget;
                const editor = getEditor(currentWidget);
                let selection = '';
                if (editor) {
                    selection = getTextSelection(editor);
                }
                else if (isMarkdownDocument(currentWidget)) {
                    selection = (_b = (_a = document.getSelection()) === null || _a === void 0 ? void 0 : _a.toString()) !== null && _b !== void 0 ? _b : '';
                }
                if (selection) {
                    return true;
                }
                return false;
            },
            isVisible: () => true,
            execute: () => {
                var _a, _b;
                const currentWidget = app.shell.currentWidget;
                const editor = getEditor(currentWidget);
                let selection = '';
                if (editor) {
                    selection = getTextSelection(editor);
                }
                else if (isMarkdownDocument(currentWidget)) {
                    selection = (_b = (_a = document.getSelection()) === null || _a === void 0 ? void 0 : _a.toString()) !== null && _b !== void 0 ? _b : '';
                }
                if (selection) {
                    codeSnippetWidget.openMetadataEditor({
                        schemaspace: CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE,
                        schema: CodeSnippetService_1.CODE_SNIPPET_SCHEMA,
                        code: selection.split('\n'),
                        onSave: codeSnippetWidget.updateMetadata
                    });
                }
            }
        });
        app.contextMenu.addItem({
            command: commandIDs.saveAsSnippet,
            selector: '.jp-Cell'
        });
        app.contextMenu.addItem({
            command: commandIDs.saveAsSnippet,
            selector: '.jp-FileEditor'
        });
        app.contextMenu.addItem({
            command: commandIDs.saveAsSnippet,
            selector: '.jp-MarkdownViewer'
        });
        const getTextSelection = (editor, markdownPreview) => {
            var _a, _b;
            const selectionObj = editor.getSelection();
            const start = editor.getOffsetAt(selectionObj.start);
            const end = editor.getOffsetAt(selectionObj.end);
            const selection = editor.model.value.text.substring(start, end);
            if (!selection && editor.model.value.text) {
                // Allow selections from a rendered notebook cell
                return (_b = (_a = document.getSelection()) === null || _a === void 0 ? void 0 : _a.toString()) !== null && _b !== void 0 ? _b : '';
            }
            return selection;
        };
        const isFileEditor = (currentWidget) => {
            return (currentWidget instanceof docregistry_1.DocumentWidget &&
                currentWidget.content instanceof fileeditor_1.FileEditor);
        };
        const isNotebookEditor = (currentWidget) => {
            return currentWidget instanceof notebook_1.NotebookPanel;
        };
        const isMarkdownDocument = (currentWidget) => {
            return currentWidget instanceof markdownviewer_1.MarkdownDocument;
        };
        const getEditor = (currentWidget) => {
            if (isFileEditor(currentWidget)) {
                const documentWidget = currentWidget;
                return documentWidget.content.editor;
            }
            else if (isNotebookEditor(currentWidget)) {
                const notebookWidget = currentWidget;
                const notebookCell = notebookWidget.content.activeCell;
                return notebookCell === null || notebookCell === void 0 ? void 0 : notebookCell.editor;
            }
        };
    }
};
exports.default = exports.code_snippet_extension;
//# sourceMappingURL=index.js.map