"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAggregateAlarmState = exports.getAlarms = exports.getCloudwatchAlarms = exports.AlarmState = void 0;
const aws_sdk_1 = require("aws-sdk");
var AlarmState;
(function (AlarmState) {
    AlarmState["OK"] = "OK";
    AlarmState["ALARM"] = "ALARM";
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
exports.getCloudwatchAlarms = async (params) => {
    const cloudwatch = new aws_sdk_1.CloudWatch();
    let nextToken;
    let results = [];
    do {
        const result = await cloudwatch
            .describeAlarms({
            AlarmTypes: ['CompositeAlarm', 'MetricAlarm'],
            ...params,
            NextToken: nextToken,
        })
            .promise();
        results = result.CompositeAlarms
            ? results.concat(result.CompositeAlarms?.map((alarm) => {
                return {
                    alarmArn: alarm.AlarmArn,
                    state: alarm.StateValue,
                    reason: alarm.StateReason,
                    alarmDescription: alarm.AlarmDescription,
                };
            }))
            : results;
        results = result.MetricAlarms
            ? results.concat(result.MetricAlarms?.map((alarm) => {
                return {
                    alarmArn: alarm.AlarmArn,
                    state: alarm.StateValue,
                    reason: alarm.StateReason,
                    alarmDescription: alarm.AlarmDescription,
                };
            }))
            : results;
        nextToken = result.NextToken;
    } while (nextToken);
    return results;
};
/**
 * Returns all alarms starting with the provided prefix in the state of ALARM. In addition,
 * any exceptions caught will be logged and converted into an alarm state of options.defaultState
 * with an associated reason.
 */
exports.getAlarms = async (searchTerms) => {
    const states = [];
    try {
        const alarmsForAccountAndRegion = await exports.getCloudwatchAlarms();
        states.push(...hasAlarmMatching(alarmsForAccountAndRegion, searchTerms).map((alarm) => {
            alarm.reason = `${alarm.alarmArn} in ${alarm.state} due to ${alarm.reason}`;
            return alarm;
        }));
        if (states.length === 0) {
            states.push({
                state: AlarmState.ALARM,
                reason: `No alarms were found for the provided search terms: ${searchTerms.join(', ')}.`,
            });
        }
    }
    catch (e) {
        const message = 'Unable to retrieve alarms.';
        states.push({ reason: message, state: AlarmState.ALARM });
        console.error(message, e);
    }
    return states;
};
/**
 * Calculate and return an aggregate alarm state across alarms returned from all
 * CloudWatch clients provided. If any client errors are thrown, the returned state
 * will be ALARM.
 *
 */
exports.getAggregateAlarmState = async (searchTerms) => {
    const alarmDetails = await exports.getAlarms(searchTerms);
    return {
        alarmDetails,
        state: alarmDetails.find((a) => a.state === AlarmState.ALARM)
            ?.state ?? AlarmState.OK,
        summary: `${alarmDetails
            .filter((a) => a.state === AlarmState.ALARM)
            .map((a) => a.reason)
            .join(', ')}`,
    };
};
const hasAlarmMatching = (alarms, searchTerms) => {
    return [
        ...new Set(alarms.flatMap((alarm) => searchTerms
            .map((searchTerm) => {
            return alarm.alarmDescription?.includes(searchTerm)
                ? alarm
                : undefined;
        })
            .filter((sameAlarm) => sameAlarm))),
    ];
};
//# sourceMappingURL=data:application/json;base64,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