# AUTOGENERATED! DO NOT EDIT! File to edit: ../003_module_query.ipynb.

# %% auto 0
__all__ = ['MatchFilter', 'AND', 'OR', 'WeakAnd', 'ANN', 'Union', 'Ranking', 'QueryProperty', 'QueryRankingFeature', 'QueryModel']

# %% ../003_module_query.ipynb 4
from typing import Optional, Dict, Callable
from fastcore.utils import patch

# %% ../003_module_query.ipynb 6
class MatchFilter(object):
    def __init__(self) -> None:    
        "Abstract class for match filters."
        pass

# %% ../003_module_query.ipynb 7
@patch
def create_match_filter(
    self: MatchFilter, 
    query: str  # Query input.
) -> str:  # Part of the YQL expression related to the filter.
    "Abstract method to be implemented that creates part of the YQL expression related to the filter."
    raise NotImplementedError

# %% ../003_module_query.ipynb 8
@patch
def get_query_properties(
    self: MatchFilter, 
    query: Optional[str] = None  # Query input.
) -> Dict:  # Contains the relevant request properties associated with the filter.
    "Abstract method to be implemented that get the relevant request properties associated with the filter."
    raise NotImplementedError

# %% ../003_module_query.ipynb 9
class AND(MatchFilter):
    def __init__(self) -> None:
        "Filter that match document containing all the query terms."
        super().__init__()

# %% ../003_module_query.ipynb 12
@patch
def create_match_filter(
    self: AND, 
    query: str  # Query input.  
) -> str:  # Part of the YQL expression related to the AND filter.
    "Creates part of the YQL expression related to the AND filter"
    return '(userInput("{}"))'.format(query)

# %% ../003_module_query.ipynb 13
@patch
def get_query_properties(
    self: AND, 
    query: Optional[str] = None  # Query input.
) -> Dict:  # Get the relevant request properties associated with the AND filter.
    "Get the relevant request properties associated with the AND filter."
    return {}


# %% ../003_module_query.ipynb 15
class OR(MatchFilter):
    def __init__(self) -> None:
        "Filter that match any document containing at least one query term."
        super().__init__()

# %% ../003_module_query.ipynb 18
@patch
def create_match_filter(
    self: OR, 
    query: str  # Query input.
) -> str:  # Part of the YQL expression related to the OR filter.
    "Creates part of the YQL expression related to the OR filter"    
    return '({{grammar: "any"}}userInput("{}"))'.format(query)

# %% ../003_module_query.ipynb 19
@patch
def get_query_properties(
    self: OR, 
    query: Optional[str] = None  # Query input.
) -> Dict:  # Get the relevant request properties associated with the OR filter.
    "Get the relevant request properties associated with the OR filter."    
    return {}

# %% ../003_module_query.ipynb 21
class WeakAnd(MatchFilter):
    def __init__(
        self, 
        hits: int,  # Lower bound on the number of hits to be retrieved. 
        field: str = "default"  # Which Vespa field to search.
    ) -> None:
        """
        Match documents according to the weakAND algorithm.

        Reference: [https://docs.vespa.ai/en/using-wand-with-vespa.html](https://docs.vespa.ai/en/using-wand-with-vespa.html)
        """
        super().__init__()
        self.hits = hits
        self.field = field

# %% ../003_module_query.ipynb 24
@patch
def create_match_filter(
    self: WeakAnd, 
    query: str  # Query input.
) -> str:  # Part of the YQL expression related to the WeakAnd filter.
    "Creates part of the YQL expression related to the WeakAnd filter"
    query_tokens = query.split(" ")
    terms = ", ".join(
        ['{} contains "{}"'.format(self.field, token) for token in query_tokens]
    )
    return '({{targetHits: {}}}weakAnd({}))'.format(self.hits, terms)


# %% ../003_module_query.ipynb 25
@patch
def get_query_properties(
    self: WeakAnd, 
    query: Optional[str] = None  # Query input.
) -> Dict:  # Get the relevant request properties associated with the WeakAnd filter.
    "Get the relevant request properties associated with the WeakAnd filter."        
    return {}

# %% ../003_module_query.ipynb 27
class ANN(MatchFilter):
    def __init__(
        self,
        doc_vector: str,  # Name of the document field to be used in the distance calculation.
        query_vector: str,  # Name of the query field to be used in the distance calculation.
        hits: int,  # Lower bound on the number of hits to return.
        label: str,  # A label to identify this specific operator instance.
        approximate: bool = True,  # True to use approximate nearest neighbor and False to use brute force. Default to True.
    ) -> None:
        """
        Match documents according to the nearest neighbor operator.

        Reference: [https://docs.vespa.ai/en/reference/query-language-reference.html](https://docs.vespa.ai/en/reference/query-language-reference.html)
        """
        super().__init__()
        self.doc_vector = doc_vector
        self.query_vector = query_vector
        self.hits = hits
        self.label = label
        self.approximate = approximate
        self._approximate = "true" if self.approximate is True else "false"

# %% ../003_module_query.ipynb 33
@patch
def create_match_filter(
    self: ANN, 
    query: str  # Query input is ignored in the ANN case.
) -> str:  # Part of the YQL expression related to the ANN filter.
    "Creates part of the YQL expression related to the ANN filter"    
    return '({{targetHits: {}, label: "{}", approximate: {}}}nearestNeighbor({}, {}))'.format(
        self.hits, self.label, self._approximate, self.doc_vector, self.query_vector
    )

# %% ../003_module_query.ipynb 34
@patch
def get_query_properties(
    self: ANN, 
    query: Optional[str] = None  # Query input is ignored in the ANN case.
) -> Dict[str, str]:  # Get the relevant request properties associated with the ANN filter.
    "Get the relevant request properties associated with the ANN filter."            
    return {}

# %% ../003_module_query.ipynb 37
class Union(MatchFilter):
    def __init__(
        self, 
        *args: MatchFilter  # Match filters to be taken the union of.
    ) -> None:
        "Match documents that belongs to the union of many match filters."
        super().__init__()
        self.operators = args

# %% ../003_module_query.ipynb 40
@patch
def create_match_filter(
    self: Union, 
    query: str  # Query input.
) -> str:  # Part of the YQL expression related to the Union filter.
    "Creates part of the YQL expression related to the Union filter"    
    match_filters = []
    for operator in self.operators:
        match_filter = operator.create_match_filter(query=query)
        if match_filter is not None:
            match_filters.append(match_filter)
    return " or ".join(match_filters)

# %% ../003_module_query.ipynb 41
@patch
def get_query_properties(
    self: Union,  # Query input. 
    query: Optional[str] = None  # Get the relevant request properties associated with the Union filter.
) -> Dict[str, str]:  # Get the relevant request properties associated with the Union filter.
    query_properties = {}
    for operator in self.operators:
        query_properties.update(operator.get_query_properties(query=query))
    return query_properties

# %% ../003_module_query.ipynb 44
class Ranking(object):
    def __init__(
        self, 
        name: str = "default",  # Name of the rank profile as defined in a Vespa search definition.
        list_features: bool = False  # Should the ranking features be returned. Either 'true' or 'false'.
    ) -> None:
        "Define the rank profile to be used during ranking."
        self.name = name
        self.list_features = "false"
        if list_features:
            self.list_features = "true"

# %% ../003_module_query.ipynb 49
class QueryProperty(object):
    def __init__(self) -> None:    
        "Abstract class for query property."
        pass    

# %% ../003_module_query.ipynb 50
@patch
def get_query_properties(
    self: QueryProperty, 
    query: Optional[str] = None  # Query input.
) -> Dict:  # Contains the relevant request properties to be included in the query.
    "Extract query property syntax."
    raise NotImplementedError


# %% ../003_module_query.ipynb 51
class QueryRankingFeature(QueryProperty):
    def __init__(
        self,
        name: str,  # Name of the feature.
        mapping: Callable[[str], List[float]],  # Function mapping a string to a list of floats.
    ) -> None:
        "Include ranking.feature.query into a Vespa query."
        super().__init__()
        self.name = name
        self.mapping = mapping

# %% ../003_module_query.ipynb 54
@patch
def get_query_properties(
    self: QueryRankingFeature, 
    query: Optional[str] = None  # Query input.
) -> Dict[str, str]:  # Contains the relevant request properties to be included in the query.
    value = self.mapping(query)
    return {"ranking.features.query({})".format(self.name): str(value)}

# %% ../003_module_query.ipynb 57
class QueryModel(object):
    def __init__(
        self,
        name: str = "default_name",  # Name of the query model. Used to tag model-related quantities, like evaluation metrics.
        query_properties: Optional[List[QueryProperty]] = None,  # Query properties to be included in the queries.
        match_phase: MatchFilter = AND(),  # Define the match criteria.
        ranking: Ranking = Ranking(),  # Define the rank criteria.
        body_function: Optional[Callable[[str], Dict]] = None,  # Function that take query as parameter and returns the body of a Vespa query.
    ) -> None:
        """
        Define a query model.

        A `QueryModel` is an abstraction that encapsulates all the relevant information
        controlling how a Vespa app matches and ranks documents.
        """
        self.name = name
        self.query_properties = query_properties if query_properties is not None else []
        self.match_phase = match_phase
        self.ranking = ranking
        self.body_function = body_function


# %% ../003_module_query.ipynb 65
@patch
def create_body(
    self: QueryModel, 
    query: str  # Query string.
) -> Dict[str, str]:  # Request body
    "Create the appropriate request body to be sent to Vespa."

    if self.body_function:
        body = self.body_function(query)
        return body

    query_properties = {}
    for query_property in self.query_properties:
        query_properties.update(query_property.get_query_properties(query=query))
    query_properties.update(self.match_phase.get_query_properties(query=query))

    match_filter = self.match_phase.create_match_filter(query=query)

    body = {
        "yql": "select * from sources * where {};".format(match_filter),
        "ranking": {
            "profile": self.ranking.name,
            "listFeatures": self.ranking.list_features,
        },
    }
    body.update(query_properties)
    return body
