# AUTOGENERATED! DO NOT EDIT! File to edit: ../002_module_evaluation.ipynb.

# %% auto 0
__all__ = ['EvalMetric', 'MatchRatio', 'Recall', 'ReciprocalRank', 'NormalizedDiscountedCumulativeGain', 'evaluate',
           'evaluate_query']

# %% ../002_module_evaluation.ipynb 4
import math
from typing import Dict, List, Union
from fastcore.utils import patch, patch_to
from pandas import DataFrame
from vespa.io import VespaQueryResponse
from vespa.application import Vespa
from vespa.query import QueryModel

# %% ../002_module_evaluation.ipynb 7
class EvalMetric(object):
    def __init__(self) -> None:
        "Abstract class for evaluation metric."
        pass

# %% ../002_module_evaluation.ipynb 8
@patch
def evaluate_query(
    self: EvalMetric,
    query_results,  # Raw query results returned by Vespa.
    relevant_docs,  # Each dict contains a doc id a optionally a doc score.
    id_field,  # The Vespa field representing the document id.
    default_score,  # Score to assign to the additional documents that are not relevant. Default to 0.
    detailed_metrics=False,  # Return intermediate computations if available.
) -> Dict:  # Metric values.
    "Abstract method to be implemented by metrics inheriting from `EvalMetric` to evaluate query results."
    raise NotImplementedError

# %% ../002_module_evaluation.ipynb 9
class MatchRatio(EvalMetric):
    def __init__(self) -> None:
        "Computes the ratio of documents retrieved by the match phase."
        super().__init__()
        self.name = "match_ratio"

# %% ../002_module_evaluation.ipynb 12
@patch
def evaluate_query(
    self: MatchRatio,
    query_results: VespaQueryResponse,  # Raw query results returned by Vespa.
    relevant_docs: List[Dict],  # Each dict contains a doc id a optionally a doc score.
    id_field: str,  # The Vespa field representing the document id.
    default_score: int,  # Score to assign to the additional documents that are not relevant. Default to 0.
    detailed_metrics=False,  # Return intermediate computations if available.
) -> Dict:  # Returns the match ratio. In addition, if `detailed_metrics=False`, returns the number of retrieved docs `_retrieved_docs` and the number of docs available in the corpus `_docs_available`.
    "Evaluate query results according to match ratio metric."
    
    retrieved_docs = query_results.number_documents_retrieved
    docs_available = query_results.number_documents_indexed
    value = 0
    if docs_available > 0:
        value = retrieved_docs / docs_available
    metrics = {
        str(self.name): value,
    }
    if detailed_metrics:
        metrics.update(
            {
                str(self.name) + "_retrieved_docs": retrieved_docs,
                str(self.name) + "_docs_available": docs_available,
            }
        )
    return metrics

# %% ../002_module_evaluation.ipynb 26
class Recall(EvalMetric):
    def __init__(
        self, 
        at: int  # Maximum position on the resulting list to look for relevant docs.
    ) -> None:
        "Compute the recall at position `at`."
        super().__init__()
        self.name = "recall_" + str(at)
        self.at = at

# %% ../002_module_evaluation.ipynb 29
@patch
def evaluate_query(
    self: Recall,
    query_results: VespaQueryResponse,  # Raw query results returned by Vespa.
    relevant_docs: List[Dict],  # Each dict contains a doc id a optionally a doc score.
    id_field: str,  # The Vespa field representing the document id.
    default_score: int,  # Score to assign to the additional documents that are not relevant. Default to 0.
    detailed_metrics=False,  # Return intermediate computations if available.
) -> Dict:  # Returns the recall value.
    """
    Evaluate query results according to recall metric.

    There is an assumption that only documents with score > 0 are relevant. Recall is equal to zero in case no
    relevant documents with score > 0 is provided.
    """

    relevant_ids = {str(doc["id"]) for doc in relevant_docs if doc.get("score", default_score) > 0}
    try:
        retrieved_ids = {
            str(hit["fields"][id_field]) for hit in query_results.hits[: self.at]
        }
    except KeyError:
        retrieved_ids = set()

    return {str(self.name): len(relevant_ids & retrieved_ids) / len(relevant_ids) if len(relevant_ids) > 0 else 0}

# %% ../002_module_evaluation.ipynb 49
class ReciprocalRank(EvalMetric):
    def __init__(
        self, 
        at: int  # Maximum position on the resulting list to look for relevant docs.
    ):
        "Compute the reciprocal rank at position `at`"
        super().__init__()
        self.name = "reciprocal_rank_" + str(at)
        self.at = at

# %% ../002_module_evaluation.ipynb 52
@patch
def evaluate_query(
    self: ReciprocalRank,
    query_results: VespaQueryResponse,  # Raw query results returned by Vespa.
    relevant_docs: List[Dict],  # Each dict contains a doc id a optionally a doc score.
    id_field: str,  # The Vespa field representing the document id.
    default_score: int,  # Score to assign to the additional documents that are not relevant. Default to 0.
    detailed_metrics=False,  # Return intermediate computations if available.
) -> Dict:  # Returns the reciprocal rank value.
    """
    Evaluate query results according to reciprocal rank metric.

    There is an assumption that only documents with score > 0 are relevant.
    """

    relevant_ids = {str(doc["id"]) for doc in relevant_docs if doc.get("score", default_score) > 0}
    rr = 0
    hits = query_results.hits[: self.at]
    for index, hit in enumerate(hits):
        try:
            if str(hit["fields"][id_field]) in relevant_ids:
                rr = 1 / (index + 1)
                break
        except KeyError:
            rr = 0

    return {str(self.name): rr}

# %% ../002_module_evaluation.ipynb 63
class NormalizedDiscountedCumulativeGain(EvalMetric):
    def __init__(
        self, 
        at: int  # Maximum position on the resulting list to look for relevant docs.
    ):
        "Compute the normalized discounted cumulative gain at position `at`."
        super().__init__()
        self.name = "ndcg_" + str(at)
        self.at = at

    @staticmethod    
    def _compute_dcg(scores: List[int]) -> float:
        return sum([score / math.log2(idx + 2) for idx, score in enumerate(scores)])        

# %% ../002_module_evaluation.ipynb 66
@patch
def evaluate_query(
    self: NormalizedDiscountedCumulativeGain,
    query_results: VespaQueryResponse,  # Raw query results returned by Vespa.
    relevant_docs: List[Dict],  # Each dict contains a doc id a optionally a doc score.
    id_field: str,  # The Vespa field representing the document id.
    default_score: int,  # Score to assign to the additional documents that are not relevant. Default to 0.
    detailed_metrics=False,  # Return intermediate computations if available.
) -> Dict:  # Returns the normalized discounted cumulative gain. In addition, if `detailed_metrics=False`, returns the ideal discounted cumulative gain `_ideal_dcg`, the discounted cumulative gain `_dcg`.
    """
    Evaluate query results according to normalized discounted cumulative gain.

    There is an assumption that documents returned by the query that are not included in the set of relevant
    documents have score equal to zero. Similarly, if the query returns a number `N < at` documents, we will
    assume that those `N - at` missing scores are equal to zero.
    """

    at = self.at
    relevant_scores = {str(doc["id"]): doc["score"] for doc in relevant_docs}
    assert default_score == 0, "NDCG default score should be zero."
    search_scores = [default_score] * at
    ideal_scores = [default_score] * at

    hits = query_results.hits[:at]
    for idx, hit in enumerate(hits):
        try:
            doc_id = str(hit["fields"][id_field])
            search_scores[idx] = relevant_scores.get(
                doc_id, default_score
            )
        except KeyError:
            search_scores[idx] = default_score

    sorted_score_list = sorted(list(relevant_scores.values()), reverse=True)[:at]
    for idx, score in enumerate(sorted_score_list):
        ideal_scores[idx] = score

    ideal_dcg = self._compute_dcg(ideal_scores)
    dcg = self._compute_dcg(search_scores)

    ndcg = 0
    if ideal_dcg > 0:
        ndcg = dcg / ideal_dcg

    metrics = {
        str(self.name): ndcg,
    }
    if detailed_metrics:
        metrics.update(
            {
                str(self.name) + "_ideal_dcg": ideal_dcg,
                str(self.name) + "_dcg": dcg,
            }
        )
    return metrics


# %% ../002_module_evaluation.ipynb 81
def _parse_labeled_data(
    df: DataFrame  # DataFrame with the following required columns ["qid", "query", "doc_id", "relevance"].
) -> List[Dict]:  # Concise representation of the labeled data, grouped by query_id and query.
    "Convert a DataFrame with labeled data to format used internally"
    required_columns = ["qid", "query", "doc_id", "relevance"]
    assert all(
        [x in list(df.columns) for x in required_columns]
    ), "DataFrame needs at least the following columns: {}".format(required_columns)
    qid_query = (
        df[["qid", "query"]].drop_duplicates(["qid", "query"]).to_dict(orient="records")
    )
    labeled_data = []
    for q in qid_query:
        docid_relevance = df[(df["qid"] == q["qid"]) & (df["query"] == q["query"])][
            ["doc_id", "relevance"]
        ]
        relevant_docs = []
        for idx, row in docid_relevance.iterrows():
            relevant_docs.append({"id": row["doc_id"], "score": row["relevance"]})
        data_point = {
            "query_id": q["qid"],
            "query": q["query"],
            "relevant_docs": relevant_docs,
        }
        labeled_data.append(data_point)
    return labeled_data

# %% ../002_module_evaluation.ipynb 84
def _evaluate_query_retry(app, flat_labeled_data, model, timeout, **kwargs):
    query_responses = app.query_batch(
        query_batch = [x[0] for x in flat_labeled_data], 
        query_model = model, 
        **{"ranking.softtimeout.enable": "false",
          "timeout": timeout},
        **kwargs)
    failed_queries = [idx for idx, x in enumerate(query_responses) if x.status_code != 200]
    count = 0
    while len(failed_queries) > 0:
        query_batch = [flat_labeled_data[idx][0] for idx in failed_queries]
        retry_query_responses = app.query_batch(
            query_batch=query_batch,
            query_model = model, 
            **{"ranking.softtimeout.enable": "false",
              "timeout": timeout},           
            **kwargs
        )
        for idx, query_response_idx in enumerate(failed_queries):
            query_responses[query_response_idx] = retry_query_responses[idx]
        failed_queries = [idx for idx, x in enumerate(query_responses) if x.status_code != 200]
        count+=1
        if count>=3: break   
    return query_responses
    
def evaluate(
    app: Vespa,  # Connection to a Vespa application.
    labeled_data: Union[List[Dict], DataFrame],  # Data containing query, query_id and relevant docs. See examples below for format.
    eval_metrics: List[EvalMetric],  # Evaluation metrics
    query_model: Union[QueryModel, List[QueryModel]],  # Query models to be evaluated
    id_field: str,  # The Vespa field representing the document id.
    default_score: int = 0,  # Score to assign to the additional documents that are not relevant.
    detailed_metrics=False,  # Return intermediate computations if available. 
    per_query=False,  # Set to True to return evaluation metrics per query.
    aggregators=None,  # Used only if `per_query=False`. List of pandas friendly aggregators to summarize per model metrics. We use ["mean", "median", "std"] by default.
    timeout=1000,  # Vespa query timeout in ms.
    **kwargs,  # Extra keyword arguments to be included in the Vespa Query.
) -> DataFrame:  # Returns query_id and metrics according to the selected evaluation metrics.
    "Evaluate a `QueryModel` according to a list of `EvalMetric`."
    
    if isinstance(labeled_data, DataFrame):
        labeled_data = _parse_labeled_data(df=labeled_data)

    if isinstance(query_model, QueryModel):
        query_model = [query_model]

    model_names = [model.name for model in query_model]
    assert len(model_names) == len(
        set(model_names)
    ), "Duplicate model names. Choose unique model names."

    evaluation = []

    for model in query_model:
        flat_labeled_data = [(x["query"], x["query_id"], x["relevant_docs"]) for x in labeled_data]
        query_responses = _evaluate_query_retry(app, flat_labeled_data, model, timeout, **kwargs)
        failed_queries = [idx for idx, x in enumerate(query_responses) if x.status_code != 200]
        if len(failed_queries) > 0:
            print(f"Failed queries for query model {model.name}: {len(failed_queries)}/{len(query_responses)}")
        timedout_queries = [idx for idx, x in enumerate(query_responses) if x.json.get("root", {}).get("errors", None) is not None]
        if len(timedout_queries) > 0:
            print(f"Timeout queries for query model {model.name}: {len(timedout_queries)}/{len(query_responses)}")
            
        for ((query, query_id, relevant_docs), query_response) in zip(flat_labeled_data, query_responses):
            evaluation_query = {"model": model.name, "query_id": query_id}
            for evaluator in eval_metrics:
                evaluation_query.update(
                    evaluator.evaluate_query(
                        query_response,
                        relevant_docs,
                        id_field,
                        default_score,
                        detailed_metrics,
                    )
                )
            evaluation.append(evaluation_query)
    evaluation = DataFrame.from_records(evaluation)
    if not per_query:
        if not aggregators:
            aggregators = ["mean", "median", "std"]
        evaluation = (
            evaluation[[x for x in evaluation.columns if x != "query_id"]]
            .groupby(by="model")
            .agg(aggregators)
            .T
        )
    return evaluation

# %% ../002_module_evaluation.ipynb 122
def evaluate_query(
    app: Vespa,  # Connection to a Vespa application.
    eval_metrics: List[EvalMetric],  # Evaluation metrics
    query_model: QueryModel,  # Query model to be evaluated  
    query_id: str,  # Query id represented as str.
    query: str,  # Query string.
    id_field: str,  # The Vespa field representing the document id.
    relevant_docs: List[Dict],  # Each dict contains a doc id a optionally a doc score.
    default_score: int = 0,  # Score to assign to the additional documents that are not relevant.
    detailed_metrics=False,  # Return intermediate computations if available.
    **kwargs,  # Extra keyword arguments to be included in the Vespa Query.
) -> Dict:  # Contains query_id and metrics according to the selected evaluation metrics.
    "Evaluate a single query according to evaluation metrics"
    
    query_results = app.query(query=query, query_model=query_model, **kwargs)
    evaluation = {"model": query_model.name, "query_id": query_id}
    for evaluator in eval_metrics:
        evaluation.update(
            evaluator.evaluate_query(
                query_results,
                relevant_docs,
                id_field,
                default_score,
                detailed_metrics,
            )
        )
    return evaluation

