"""Global options for ITables.

These parameters are documented at
https://mwouts.github.io/itables/advanced_parameters.html
"""

from pathlib import Path
from typing import Literal, Union

import itables.typing as typing
import itables.utils as utils

__non_options = set()
__non_options = set(locals())

"""Table layout, see https://datatables.net/reference/option/layout
NB: to remove a control, replace it by None"""
layout: dict[str, Union[str, None]] = {
    "topStart": "pageLength",
    "topEnd": "search",
    "bottomStart": "info",
    "bottomEnd": "paging",
}

"""Show the index? Possible values: True, False and 'auto'. In mode 'auto', the index is not shown
if it has no name and its content is range(N)"""
showIndex: Literal[True, False, "auto"] = "auto"

"""Default DataTables classes. See https://datatables.net/manual/styling/classes"""
classes: Union[str, list[str]] = "display nowrap"

"""Default table style. Use
- 'table-layout:auto' to compute the layout automatically
- 'width:auto' to fit the table width to its content
- 'margin:auto' to center the table.
Combine multiple options using ';'.

NB: When scrollX=true, "margin:auto" is replaced with "margin:0"
to avoid an issue with misaligned headers
"""
style: Union[str, dict[str, str]] = (
    "table-layout:auto;width:auto;margin:auto;caption-side:bottom"
)

"""Maximum bytes before downsampling a table"""
maxBytes: Union[str, int] = "64KB"

"""Maximum number of rows or columns before downsampling a table"""
maxRows: int = 0
maxColumns: int = 200

"""By default we don't sort the table"""
order: list[str] = []

"""Authorize, or not, the use of HTML in the table content.

Make sure that you trust the content of your tables before
setting this option to True.

NOTE: This option does not apply to Pandas Styler objects.
For the Style objects the HTML generated by Pandas is used as is,
so you need to make sure that you trust their content.
"""
allow_html: bool = False

"""Optional table footer"""
footer: bool = False

"""Column filters"""
column_filters: Literal[False, "header", "footer"] = False

"""Should a warning appear when we have to encode an unexpected type?"""
warn_on_unexpected_types: bool = True

"""Should a warning appear when the selection targets rows that have been
filtered by the downsampling?"""
warn_on_selected_rows_not_rendered: bool = True

"""The DataTables URL for the connected mode"""
dt_url: str = utils.UNPKG_DT_BUNDLE_URL

"""The DataTable bundle for the offline mode
(this option is for 'init_notebook_mode')"""
dt_bundle: Union[str, Path] = utils.find_package_file("html/dt_bundle.js")

"""Display the ITables animated logo when loading"""
display_logo_when_loading: bool = True

"""Make the text in the table header selectable. When False, clicking
on the column header will sort the table. See #227"""
text_in_header_can_be_selected: bool = True

"""Check that all options passed to ITable are valid and have the expected type."""
warn_on_undocumented_option: bool = typing.is_typeguard_available()
if warn_on_undocumented_option:
    typing.check_itable_arguments(
        {k: v for k, v in locals().items() if k not in __non_options},
        typing.ITableOptions,
    )
