import os
from os import PathLike
from pathlib import Path
from functools import lru_cache
import requests

here = Path(__file__).parent

def get_openrouter_api_key() -> str:
    """Retrieve the OpenRouter API key from environment variables."""
    api_key = os.getenv('OPENROUTER_API_KEY')
    if not api_key:
        raise ValueError("Please set the OPENROUTER_API_KEY environment variable.")
    return api_key


def list_openrouter_models(api_key:str) -> list[str]:
    """get the list of model names from OpenRouter API"""
    models = _get_openrouter_models(api_key)
    return [model['id'] for model in models]

@lru_cache()
def _get_openrouter_models(api_key: str) -> list[dict]:
    """Get the list of models (including all metadata) from OpenRouter API"""
    url = "https://openrouter.ai/api/v1/models"
    headers = {"Authorization": f"Bearer {api_key}"}
    response = requests.get(url, headers=headers)
    if response.status_code != 200:
        raise Exception(f"Error: {response.status_code}")

    data = response.json()['data']
    return data


def _create_models_types_file(api_key:str|None = None, file:PathLike = here / 'openrouter_models.py'):
    """
    Dev helper tool: create a file with a type declaration `ModelName` containing all OpenRouter models.
    """
    if api_key is None: api_key = get_openrouter_api_key()
    file = Path(file)
    models = _get_openrouter_models(api_key)
    models.sort(key=lambda x: x['id'])
    model_names = [model['id'] for model in models]
    context_lengths = [model['context_length'] for model in models]
    supports_tools = ["tools" in model['supported_parameters'] for model in models]


    print(f"Creating {file.relative_to(here.parent)} with {len(models)} models")
    name_lines = ',\n    '.join(f"'{model_name}'" for model_name in model_names)
    attributes_lines = ',\n    '.join(f'''{f'"{model_name}":':<40}Attr(context_size={context_length}, supports_tools={supports_tool})''' for model_name, context_length, supports_tool in zip(model_names, context_lengths, supports_tools))
    file.write_text(f'''\
# DO NOT EDIT THIS FILE MANUALLY
# This file is generated by calling `_create_models_types_file()` in openrouter_utils.py

from typing import Literal
from dataclasses import dataclass


ModelName = Literal[
    {name_lines}
]


@dataclass
class Attr:
    context_size: int
    supports_tools: bool
    # TBD: may add more in the future


attributes_map: dict[ModelName, Attr] = {{
    {attributes_lines}
}}
''')