"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination
 */
class LambdaDestination {
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload') // Extract response payload
                })
            ]
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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