use super::snapshot::PyDbState;

use super::base_env::BaseEnv;
use super::snapshot;
use crate::types::{PyAddress, PyEvent, PyExecutionResult, PyRevertError};
use pyo3::exceptions::PyRuntimeError;
use pyo3::prelude::*;
use pyo3::types::PyBytes;
use verbs_rs::LocalDB;

/// Simulation environment initialised with an empty in-memory database
///
/// Wraps an EVM and in-memory db along with additional functionality
/// for simulation updates and event tracking. This environment can
/// also be initialised from a snapshot to speed up simulation
/// initialisation.
///
/// Examples
/// --------
///
/// .. code-block:: python
///
///    # Initialise a completely empty db
///    env = EmptyEnv(101)
///    # Or initialise from a snapshot
///    env = EmptyEnv(101, snapshot=snapshot)
///    # Or load a cache from a previous forked run
///    env = EmptyEnv(101, cache=cache)
///    ...
///    env.submit_call(...)
///
#[pyclass]
pub struct EmptyEnv(BaseEnv<LocalDB>);

#[pymethods]
impl EmptyEnv {
    #[new]
    #[pyo3(signature = (seed, snapshot=None, cache=None))]
    pub fn new(
        seed: u64,
        snapshot: Option<PyDbState>,
        cache: Option<snapshot::PyRequests>,
    ) -> PyResult<Self> {
        match (snapshot, cache) {
            (None, None) => Ok(Self(BaseEnv::<LocalDB>::new(0, 0, seed))),
            (None, Some(c)) => Ok(Self(BaseEnv::<LocalDB>::from_cache(seed, c))),
            (Some(s), None) => Ok(Self(BaseEnv::<LocalDB>::from_snapshot(seed, s))),
            (Some(_), Some(_)) => Err(PyRuntimeError::new_err(
                "Env must be initialised from either a snapshot or a cache but not both.",
            )),
        }
    }

    /// Export a snap shot of the EVM state and block parameters
    ///
    /// Creates a copy of the EVM storage and state of the current block in a
    /// format that can be exported to Python. This snapshot can then be used
    /// to initialise new simulation environments.
    pub fn export_snapshot<'a>(&mut self, py: Python<'a>) -> PyResult<snapshot::PyDbState<'a>> {
        Ok(self.0.export_state(py))
    }

    /// Current step (i.e. block) of the simulation
    ///
    /// Returns
    /// -------
    /// int
    ///     Current step of the simulation.
    ///
    #[getter]
    fn get_step(&self) -> PyResult<usize> {
        Ok(self.0.step)
    }

    /// Process the next block in the simulation
    ///
    /// Update the state of the simulation by processing the next
    /// simulated block. This performs several steps:
    ///
    /// * Update the simulated time and block number
    /// * Sort the queue of calls submitted by agents
    /// * Process the queue of calls, updating the state of the EVM
    /// * Store any events generated by the transactions in this block
    ///
    pub fn process_block(&mut self) -> PyResult<()> {
        self.0.process_block();
        Ok(())
    }

    /// Get a list of events/logs generated in the last block
    ///
    /// Returns a list of events/logs generated in the last block.
    /// Events are a tuple containing:
    ///
    /// * The selector of the function called
    /// * A vector of logs
    /// * The step the event was generated
    /// * The order the event was created inside a block
    ///
    /// Returns
    /// -------
    /// list[tuple]
    ///     List of events
    ///
    pub fn get_last_events<'a>(&'a mut self, py: Python<'a>) -> PyResult<Vec<PyEvent>> {
        Ok(self.0.get_last_events(py))
    }

    /// Returns a list of events/logs generated over the
    /// course of the simulation.
    /// Events are a tuple containing:
    ///
    /// * Boolean indicating if the transaction was successful
    /// * The selector of the function called
    /// * A vector of logs
    /// * The step the event was generated
    /// * The order the event was created inside a block
    ///
    /// Returns
    /// -------
    /// list[tuple]
    ///     List of events
    ///
    pub fn get_event_history<'a>(&'a mut self, py: Python<'a>) -> PyResult<Vec<PyEvent>> {
        Ok(self.0.get_event_history(py))
    }

    /// submit_transaction(sender: bytes,  transact_to: bytes, encoded_args: bytes, value: int, checked: bool)
    ///
    /// Submit a call into the next block
    ///
    /// Submit a transaction into the queue to be processed
    /// in the next block. Each simulation step agents submit
    /// calls which are then shuffled and processed to update
    /// the EVM state.
    ///
    /// Parameters
    /// ----------
    /// sender: bytes
    ///     Byte encoded address of the transaction sender.
    /// transact_to: bytes
    ///     Byte encoded address of the contract to call.
    /// encoded_args: bytes
    ///     ABI encoded function selector and arguments.
    /// value: int
    ///     Value attached to the transaction.
    /// checked: bool
    ///     If ``True`` the simulation will halt if this transaction
    ///     is reverted.
    ///
    pub fn submit_transaction(
        &mut self,
        sender: PyAddress,
        transact_to: PyAddress,
        encoded_args: Vec<u8>,
        value: u128,
        checked: bool,
    ) -> PyResult<()> {
        self.0
            .submit_transaction(sender, transact_to, encoded_args, value, checked);
        Ok(())
    }

    /// submit_transactions(transactions: list[tuple])
    ///
    /// Submit a list of transactions into the next block
    ///
    /// Submit a list of transaction into the queue to be processed
    /// in the next block. Each simulation step agents submit
    /// calls which are then shuffled and processed to update
    /// the EVM state.
    ///
    /// Parameters
    /// ----------
    /// transactions: list[tuple[bytes, bytes, bytes, int, bool]]
    ///     List of transactions, where a transaction is a tuple
    ///     containing:
    ///
    ///     * The byte encoded address of the sender
    ///     * The byte encoded address of the contract
    ///     * The ABI byte encoded arguments and function selector
    ///     * The value attached to the transaction
    ///     * Flag if ``True`` means the simulation will halt if this
    ///       transaction fails
    ///
    pub fn submit_transactions(
        &mut self,
        transactions: Vec<(PyAddress, PyAddress, Vec<u8>, u128, bool)>,
    ) -> PyResult<()> {
        self.0.submit_transactions(transactions);
        Ok(())
    }

    /// deploy_contract(deployer: bytes, contract_name: str, bytecode: bytes) -> bytes
    ///
    ///Deploy a contract
    ///
    /// Deploys a contract to the EVM by calling the constructor.
    ///
    /// Parameters
    /// ----------
    /// deployer: bytes
    ///     Byte encoded address of the deployer.
    /// contract_name: str
    ///     Name of the contract to deploy, only used for
    ///     logging/debugging purposes.
    /// bytecode: bytes
    ///     Contract deployment bytecode and ABI encoded
    ///     constructor arguments.
    ///
    /// Returns
    /// -------
    /// bytes
    ///     Byte encoded address that contract is deployed to.
    ///
    pub fn deploy_contract<'a>(
        &mut self,
        py: Python<'a>,
        deployer: PyAddress,
        contract_name: &str,
        bytecode: Vec<u8>,
    ) -> PyResult<&'a PyBytes> {
        Ok(PyBytes::new(
            py,
            self.0
                .deploy_contract(deployer, contract_name, bytecode)
                .as_slice(),
        ))
    }

    /// create_account(address: bytes, start_balance: int)
    ///
    /// Create an account
    ///
    /// Create a new account with balance of ETH.
    ///
    /// Parameters
    /// ----------
    /// address: bytes
    ///     Address to deploy account to
    /// start_balance: int
    ///     Starting ETH balance of the account (in wei)
    ///
    pub fn create_account(&mut self, address: PyAddress, start_balance: u128) {
        self.0.create_account(address, start_balance)
    }

    /// call(sender: bytes, contract_address: bytes, encoded_args: bytes, value: int) -> tuple[bytes, list, int]
    ///
    /// Directly call the EVM
    ///
    /// Call the EVM and return the result and events. This
    /// does not update the state of the EVM.
    ///
    /// Parameters
    /// ----------
    /// sender: bytes
    ///     Address of the transaction sender.
    /// contract_address: bytes
    ///     Address of the contract to call.
    /// encoded_args: bytes
    ///     ABI encoded function selector and arguments
    /// value: int
    ///     Value attached to this transaction.
    ///
    /// Returns
    /// -------
    /// tuple[bytes, list[tuple], int]
    ///     Tuple containing optional, byte-encoded results
    ///     of the transaction, and list of logs generated by
    ///     the transaction.
    ///
    /// Raises
    /// ------
    /// verbs.envs.RevertError
    ///     Raises an exception if the transaction is reverted.
    ///
    pub fn call<'a>(
        &'a mut self,
        py: Python<'a>,
        sender: PyAddress,
        contract_address: PyAddress,
        encoded_args: Vec<u8>,
        value: u128,
    ) -> PyResult<PyExecutionResult> {
        let result = self
            .0
            .call(py, sender, contract_address, encoded_args, value);
        match result {
            Ok(x) => Ok(x),
            Err(x) => Err(PyRevertError::new_err(x.output)),
        }
    }

    /// execute(sender: bytes, contract_address: bytes, encoded_args: bytes, value: int) -> tuple[bytes, list, int]
    ///
    ///Directly execute a transaction
    ///
    /// Execute a transaction and return the result and events.
    /// This update the state of the EVM.
    ///
    /// Parameters
    /// ----------
    /// sender: bytes
    ///     Address of the transaction sender.
    /// contract_address: bytes
    ///     Address of the contract to call.
    /// encoded_args: bytes
    ///     ABI encoded function selector and arguments
    /// value: int
    ///     Value attached to the transaction
    ///
    /// Returns
    /// -------
    /// tuple[bytes, list[tuple]]
    ///     Tuple containing optional, byte-encoded results
    ///     of the transaction, and list of logs generated by
    ///     the transaction.
    ///
    /// Raises
    /// ------
    /// verbs.envs.RevertError
    ///     Raises an exception if the transaction is reverted.
    ///
    pub fn execute<'a>(
        &'a mut self,
        py: Python<'a>,
        sender: PyAddress,
        contract_address: PyAddress,
        encoded_args: Vec<u8>,
        value: u128,
    ) -> PyResult<PyExecutionResult> {
        let result = self
            .0
            .execute(py, sender, contract_address, encoded_args, value);
        match result {
            Ok(x) => Ok(x),
            Err(x) => Err(PyRevertError::new_err(x.output)),
        }
    }
}
