//! Transaction related data types

use alloy_primitives::{Address, U256};
use alloy_sol_types::SolCall;
use revm::primitives::{Log, Output};

/// EVM transaction argument data
#[derive(Clone)]
pub struct Transaction {
    /// 4 byte function selector
    pub function_selector: [u8; 4],
    /// Address of the contract caller
    pub callee: Address,
    /// Address of the contract to call
    pub transact_to: Address,
    /// ABI encoded arguments bytes
    pub args: Vec<u8>,
    /// Value attached to the transaction
    pub value: U256,
    /// Flag, if `true` the simulation will halt (panic)
    /// if this transaction is reverted.
    pub checked: bool,
}

impl Transaction {
    pub fn new<T: SolCall>(
        callee: Address,
        contract: Address,
        args: T,
        value: U256,
        checked: bool,
    ) -> Self {
        Transaction {
            function_selector: T::SELECTOR,
            callee,
            transact_to: contract,
            args: args.abi_encode(),
            value,
            checked,
        }
    }
}

/// Result of a transaction included any generated events
pub struct TransactionResult {
    /// Flag whether transaction was successful
    pub success: bool,
    /// Output data
    pub output: Output,
    /// Vec of events returned by the transaction
    pub events: Option<Event>,
}

/// Wraps event logs with additional event information
///
/// Events are created for each transaction processed
/// during the simulation, included failed (reverted)
/// transactions. This allow a history of simulation
/// events to be recreated after a simulation
///
pub struct Event {
    /// If the event was successful (i.e. `false`
    /// indicates a transaction was reverted)
    pub success: bool,
    /// 4 byte function selector of the called function
    pub function_selector: [u8; 4],
    /// Event data generated by the transaction
    pub logs: Vec<Log>,
    /// Simulation step the event was created
    pub step: usize,
    /// Sequence the event was created inside a block
    pub sequence: usize,
}
