import inspect
import json
import re
import sys
import traceback
from dataclasses import dataclass
from inspect import _empty
from typing import Any, Dict, List

import numpy as np
from dataclasses_json import dataclass_json

# whether to hide metaclasses which accept other classes as inputs
HIDE_META_CLASSES = True
HIDDEN_PARAMETERS = [
    "n_jobs",
    "random_state",
    "verbose",
    "dtype",
    "pooling_func",
    "score_func",
]
JS_FOLDER = "../../../../../ui/src/editor/components/autogenerated/"
JS_SCHEMA_FOLDER = "../../../../../ui/src/editor/components/schemas/"


def try_get_name(value):
    try:
        return value.__name__
    except:
        return None


def convert_value_to_safe_js(name, value):
    # overrides
    # @TODO fix this
    if name in ["missing_values", "loss"]:
        return None

    if type(value) == float or type(value) == int:
        return value
    if value is None or value == "None" or value == np.nan or str(value) == "nan":
        return None
    elif isinstance(value, tuple):
        return json.dumps(value)
    elif isinstance(value, str) and isfloat(value):
        return float(value)
    elif value == "True" or value == True:
        return True
    elif value == "False" or value == False:
        return False
    elif isinstance(value, float) or isinstance(value, int):
        return value
    elif try_get_name(value) == "_empty":
        return None
    elif isinstance(value, str):
        value = value.strip("'").strip('"')
        return f"{value}"
    else:
        return str(value)


def isfloat(value):
    try:
        _ = float(value)
        return True
    except:
        return False


def get_param_names(cls):
    """Get parameter names for the estimator"""
    # fetch the constructor or the original constructor before
    # deprecation wrapping if any
    init = getattr(cls.__init__, "deprecated_original", cls.__init__)
    if init is object.__init__:
        # No explicit constructor to introspect
        return []

    # introspect the constructor arguments to find the model parameters
    # to represent
    init_signature = inspect.signature(init)
    # Consider the constructor parameters excluding 'self'
    parameters = [
        p
        for p in init_signature.parameters.values()
        if p.name != "self" and p.kind != p.VAR_KEYWORD
    ]
    for p in parameters:
        if p.kind == p.VAR_POSITIONAL:
            raise RuntimeError(
                "scikit-learn estimators should always "
                "specify their parameters in the signature"
                " of their __init__ (no varargs)."
                " %s with constructor %s doesn't "
                " follow this convention." % (cls, init_signature)
            )
    # Extract and sort argument names excluding 'self'
    return [p.name for p in parameters if p.name not in HIDDEN_PARAMETERS]


def is_meta_class(parameters):
    for param_name in ["base_estimator", "estimators"]:
        if param_name in parameters:
            return True
    return False


def filter_estimator(cls):
    return cls is not None and not is_meta_class(cls["params_list"])


def try_numeric(s):
    for func in [int, float, str]:
        try:
            return func(s)
        except:
            continue
    return s


class Element:
    def __init__(self, name, module, params, doc, obj_type):
        self.name = name
        self.module = module
        self.params = params
        self.doc = doc
        self.obj_type = obj_type

    def __repr__(self):
        return self.name


@dataclass_json
@dataclass
class Param:
    name: str
    types: List
    choices: List
    is_optional: bool
    default: Any

    @property
    def default_js(self):
        return convert_value_to_safe_js(self.name, self.default)

    @property
    def control_js(self):
        # return "StrControl"
        if len(self.types) == 0:
            return "StrControl"
        elif len(self.choices) > 0:
            return "DropDownControl"
        elif self.types[0] == "int":
            return "IntControl"
        elif self.types[0] == "float":
            return "FloatControl"
        elif self.types[0] == "bool":
            return "BoolControl"
        else:
            return "StrControl"


@dataclass
class Input:
    key: str
    name: str
    socket: str
    many_connections: bool

    def many_connections_js(self):
        return "true" if self.many_connections else "false"


@dataclass
class Output:
    key: str
    name: str
    socket: str
    many_connections: bool

    def many_connections_js(self):
        return "true" if self.many_connections else "false"


def extract_default(s):
    try:
        return re.findall("default=([^\)]*)", s)[0]
    except:
        return None


def enumerate_types(s):
    types = []
    for t in ["int", "float", "string", "bool"]:
        if t in s:
            types.append(t)
    return types


def extract_choices(s):
    raw_choices = re.findall("{(.*?)}", s)
    if raw_choices:
        return [c.strip("""'", """) for c in raw_choices[0].split(",")]
    else:
        return []


def deduct_parameter_types(params, doc, class_name, fit_details: inspect.Signature):
    output = []
    default_values_dict = {
        name: param.default for name, param in fit_details.parameters.items()
    }
    print("Doc", doc)
    for param in params:
        try:
            details = re.findall(f"{param}.?:.?(.*)\n", doc)[0]
        except:
            # LightGBM parameters are described in parent's class init function
            details = re.findall(
                f"{param}.?:.?(.*)\n", class_name.__bases__[0].__init__.__doc__
            )[0]
        default = default_values_dict.get(param)
        choices = extract_choices(details)
        types = enumerate_types(details)
        is_optional = "optional" in details

        if param == "ngram_range":
            param = Param(
                name=param,
                types=["intrange"],
                choices=[],
                is_optional=True,
                default=convert_value_to_safe_js(param, default),
            )
        else:
            param = Param(
                name=param,
                types=types,
                choices=choices,
                is_optional=is_optional,
                default=convert_value_to_safe_js(param, default),
            )
        output.append(param)
    return output


def classify_sklearn_class_type(cls):
    methods = dir(cls)
    name = cls.__name__
    if "predict_proba" in methods:
        return "classifier"
    elif "score" in methods or "regressor" in name.lower():
        return "regressor"
    elif "predict" in methods:
        return "other classifier"
    elif "transform" in methods:
        return "transformer"
    else:
        return "other"


def get_input_list(cls):
    if hasattr(cls, "fit"):
        return [
            Input(inp, inp, "anySocket", True)
            for inp in inspect.signature(cls.fit).parameters
            if inp != "self"
        ]
    elif hasattr(cls, "run"):
        return [
            Input(inp, inp, "anySocket", True)
            for inp in inspect.signature(cls.run).parameters
            if inp != "self" and inp != "context"
        ]


def get_output_list(cls):
    outputs = []
    # model is a symbol (not the actual method)
    if hasattr(cls, "predict"):
        outputs.append(("model", "model"))
    if hasattr(cls, "predict"):
        outputs.append(("predict", "predictions"))
    if hasattr(cls, "predict_proba"):
        outputs.append(("predict_proba", "probabilities"))
    if hasattr(cls, "transform"):
        outputs.append(("transform", "transformed"))
    return [
        Output(out_key, out_name, "anySocket", True) for out_key, out_name in outputs
    ]


def extract_class_info(v):
    try:
        doc = v["cls"].__doc__
        params_list = get_param_names(v["cls"])
        init_params = inspect.signature(v["cls"].__init__)
        print(init_params)
        print(v["cls"])
        return dict(
            name=v["name"],
            js_name=v.get("prefix", "") + v["name"],
            module=v["submodule"],
            py_module=v["submodule"],
            js_component_prefix=v.get("prefix", ""),
            human_module=".".join(
                [
                    m
                    for m in v.get("human_submodule", v["submodule"]).split(".")
                    if not m.startswith("_")
                ]
            ),
            params_list=params_list,
            params=deduct_parameter_types(params_list, doc, v["cls"], init_params),
            doc=doc,
            obj_type=classify_sklearn_class_type(v["cls"]),
            # params_details=inspect.signature(v["cls"]().__init__),
            # fit_details=inspect.signature(v["cls"]().fit),
            input_list=get_input_list(v["cls"]),
            output_list=get_output_list(v["cls"]),
        )
    except Exception as e:
        # print("error", k, e)
        exc_type, exc_value, exc_traceback = sys.exc_info()
        traceback.print_exception(
            exc_type, exc_value, exc_traceback, limit=2, file=sys.stdout
        )
        return None


def filter_class(v):
    return ("run" in dir(v["cls"])) or (
        "tests" not in v["submodule"]
        and ("predict" in dir(v["cls"]) or "transform" in dir(v["cls"]))
        and "get_params" in dir(v["cls"])
        and v["cls"].__doc__ is not None
    )
