"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnumType = exports.UnionType = exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * (experimental) Interface Types are abstract types that includes a certain set of fields that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
        this.directives = props.directives;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Intermediate Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * (experimental) Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
            directives: props.directives,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * (experimental) Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.generateResolver(options.fieldName, options.field.fieldOptions);
        this.definition[options.fieldName] = options.field;
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Generate the resolvers linked to this Object Type.
     *
     * @experimental
     */
    generateResolver(fieldName, options) {
        if (!(options === null || options === void 0 ? void 0 : options.dataSource))
            return;
        if (!this.resolvers) {
            this.resolvers = [];
        }
        this.resolvers.push(options.dataSource.createResolver({
            typeName: this.name,
            fieldName: fieldName,
            pipelineConfig: options.pipelineConfig,
            requestMappingTemplate: options.requestMappingTemplate,
            responseMappingTemplate: options.responseMappingTemplate,
        }));
    }
}
exports.ObjectType = ObjectType;
/**
 * (experimental) Input Types are abstract types that define complex objects.
 *
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Input Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this input type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InputType = InputType;
/**
 * (experimental) Union Types are abstract types that are similar to Interface Types, but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
class UnionType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((def) => this.addField({ field: def.attribute() }));
    }
    /**
     * (experimental) Create a GraphQL Type representing this Union Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this Union type.
     *
     * @experimental
     */
    toString() {
        // Return a string that appends all Object Types for this Union Type
        // i.e. 'union Example = example1 | example2'
        return Object.values(this.definition).reduce((acc, field) => `${acc} ${field.toString()} |`, `union ${this.name} =`).slice(0, -2);
    }
    /**
     * (experimental) Add a field to this Union Type.
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        var _a;
        if (options.fieldName) {
            throw new Error('Union Types cannot be configured with the fieldName option. Use the field option instead.');
        }
        if (!options.field) {
            throw new Error('Union Types must be configured with the field option.');
        }
        if (options.field && !(options.field.intermediateType instanceof ObjectType)) {
            throw new Error('Fields for Union Types must be Object Types.');
        }
        this.definition[((_a = options.field) === null || _a === void 0 ? void 0 : _a.toString()) + 'id'] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.UnionType = UnionType;
/**
 * (experimental) Enum Types are abstract types that includes a set of fields that represent the strings this type can create.
 *
 * @experimental
 */
class EnumType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((fieldName) => this.addField({ fieldName }));
    }
    /**
     * (experimental) Create an GraphQL Type representing this Enum Type.
     *
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this enum type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'enum',
            name: this.name,
            fields: Object.keys(this.definition),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Enum Type.
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (options.field) {
            throw new Error('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
        }
        if (!options.fieldName) {
            throw new Error('When adding a field to an Enum Type, you must configure the fieldName option.');
        }
        if (options.fieldName.indexOf(' ') > -1) {
            throw new Error(`Enum Type values cannot have whitespace. Received: ${options.fieldName}`);
        }
        this.definition[options.fieldName] = schema_field_1.GraphqlType.string();
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.EnumType = EnumType;
//# sourceMappingURL=data:application/json;base64,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