import { AuthorizationType } from './graphqlapi';
import { Directive } from './schema-base';
import { InterfaceType } from './schema-intermediate';
/**
 * Utility enum for Schema class
 */
export declare enum SchemaMode {
    FILE = "FILE",
    CODE = "CODE"
}
/**
 * Generates an addition to the schema
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
export interface SchemaAdditionOptions {
    /**
     * the prefix for this additon (type, interface, enum, input, schema)
     */
    readonly prefix: string;
    /**
     * the name for this addition (some additions dont need this [i.e. schema])
     *
     * @default - no name
     */
    readonly name?: string;
    /**
     * the interface types if this is creating an object type
     *
     * @default - no interfaces
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * the directives for this type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
    /**
     * the fields to reduce onto the addition
     */
    readonly fields: string[];
    /**
     * the authorization modes for this graphql type
     */
    readonly modes?: AuthorizationType[];
}
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
export declare function shapeAddition(options: SchemaAdditionOptions): string;
/**
 * Utility class to represent DynamoDB key conditions.
 */
export declare abstract class BaseKeyCondition {
    and(cond: BaseKeyCondition): BaseKeyCondition;
    renderExpressionNames(): string;
    renderExpressionValues(): string;
    abstract renderCondition(): string;
    abstract keyNames(): string[];
    abstract args(): string[];
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
export declare class BeginsWith extends BaseKeyCondition {
    private readonly keyName;
    private readonly arg;
    constructor(keyName: string, arg: string);
    renderCondition(): string;
    keyNames(): string[];
    args(): string[];
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
export declare class BinaryCondition extends BaseKeyCondition {
    private readonly keyName;
    private readonly op;
    private readonly arg;
    constructor(keyName: string, op: string, arg: string);
    renderCondition(): string;
    keyNames(): string[];
    args(): string[];
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
export declare class Between extends BaseKeyCondition {
    private readonly keyName;
    private readonly arg1;
    private readonly arg2;
    constructor(keyName: string, arg1: string, arg2: string);
    renderCondition(): string;
    keyNames(): string[];
    args(): string[];
}
