"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_1 = require("@aws-cdk/core");
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for concrete datasources
 *
 * @experimental
 */
class BaseDataSource extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        var _a, _b;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        const name = (_a = props.name) !== null && _a !== void 0 ? _a : id;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: name,
            description: props.description,
            serviceRoleArn: (_b = this.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            ...extended,
        });
        this.name = name;
        this.api = props.api;
    }
    /**
     * (experimental) creates a new resolver for this datasource and API using the given properties.
     *
     * @experimental
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for resource backed datasources
 *
 * @experimental
 */
class BackedDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * (experimental) An AppSync dummy datasource.
 *
 * @experimental
 */
class NoneDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * (experimental) An AppSync datasource backed by a DynamoDB table.
 *
 * @experimental
 */
class DynamoDbDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * (experimental) An AppSync datasource backed by a http endpoint.
 *
 * @experimental
 */
class HttpDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const authorizationConfig = props.authorizationConfig ? {
            authorizationType: 'AWS_IAM',
            awsIamConfig: props.authorizationConfig,
        } : undefined;
        super(scope, id, props, {
            type: 'HTTP',
            httpConfig: {
                endpoint: props.endpoint,
                authorizationConfig,
            },
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * (experimental) An AppSync datasource backed by a Lambda function.
 *
 * @experimental
 */
class LambdaDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
//# sourceMappingURL=data:application/json;base64,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