# Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"). You
# may not use this file except in compliance with the License. A copy of
# the License is located at
#
# http://aws.amazon.com/apache2.0/
#
# or in the "license" file accompanying this file. This file is
# distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF
# ANY KIND, either express or implied. See the License for the specific
# language governing permissions and limitations under the License.

from ..lib import ec2, utils
from ..resources.strings import strings
from ..resources.statics import namespaces, option_names

class CreateEnvironmentRequest(object):

    def __init__(self, app_name=None, env_name=None, cname=None, platform=None,
                 tier=None, instance_type=None, version_label=None,
                 instance_profile=None, service_role=None,
                 single_instance=False, key_name=None,
                 sample_application=False, tags=None, scale=None,
                 database=None, vpc=None, template_name=None, group_name=None,
                 elb_type=None, platform_arn=None):
        self.app_name = app_name
        self.cname = cname
        self.env_name = env_name
        self.instance_profile = instance_profile
        self.instance_type = instance_type
        self.key_name = key_name
        self.platform = platform
        self.sample_application = sample_application
        self.service_role = service_role
        self.single_instance = single_instance
        self.template_name = template_name
        self.tier = tier
        self.version_label = version_label
        self.platform_arn = platform_arn
        self.group_name = group_name
        if tags is None:
            self.tags = []
        else:
            self.tags = list(tags)
        if database is None:
            self.database = {}
        else:
            self.database = dict(database)
        if vpc is None:
            self.vpc = {}
        else:
            self.vpc = dict(vpc)

        self.elb_type = elb_type
        self.scale = None
        self.option_settings = []
        self.compiled = False
        self.description = strings['env.description']

        if not self.app_name:
            raise TypeError(self.__class__.__name__ + ' requires key-word argument app_name')
        if not self.env_name:
            raise TypeError(self.__class__.__name__ + ' requires key-word argument env_name')

        if scale:
            if not isinstance(scale, int):
                raise TypeError('key-word argument scale must be of type int')
            else:
                self.scale = str(scale)

    def __eq__(self, other):
        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return self.__dict__ != other.__dict__

    def add_option_setting(self, namespace, option_name, value, resource=None):
        setting = {'Namespace': namespace,
                   'OptionName': option_name,
                   'Value': value}
        if resource:
            setting['ResourceName'] = resource

        self.option_settings.append(setting)

    def convert_to_kwargs(self):
        self.compile_option_settings()
        return self.get_standard_kwargs()

    def compile_option_settings(self):
        if not self.compiled:
            self.add_client_defaults()
            self.compile_database_options()
            self.compile_vpc_options()
            self.compile_common_options()
            self.compiled = True

    def get_standard_kwargs(self):
        kwargs = {
            'ApplicationName': self.app_name,
            'EnvironmentName': self.env_name,
            'OptionSettings': self.option_settings,
            }
        if self.platform:
            kwargs['SolutionStackName'] = self.platform.name
        if self.platform_arn:
            kwargs['PlatformArn'] = self.platform_arn
        if self.description:
            kwargs['Description'] = self.description
        if self.cname:
            kwargs['CNAMEPrefix'] = self.cname
        if self.template_name:
            kwargs['TemplateName'] = self.template_name
        if self.version_label:
            kwargs['VersionLabel'] = self.version_label
        if self.tags:
            kwargs['Tags'] = self.tags
        if self.tier:
            kwargs['Tier'] = self.tier.to_struct()

        if self.scale:
            self.add_option_setting(
                namespaces.AUTOSCALING,
                option_names.MAX_SIZE,
                self.scale)
            self.add_option_setting(
                namespaces.AUTOSCALING,
                option_names.MIN_SIZE,
                self.scale)

        return kwargs

    def compile_common_options(self):
        if self.instance_profile:
            self.add_option_setting(
                namespaces.LAUNCH_CONFIGURATION,
                option_names.IAM_INSTANCE_PROFILE,
                self.instance_profile)
        if self.service_role:
            self.add_option_setting(
                namespaces.ENVIRONMENT,
                option_names.SERVICE_ROLE,
                self.service_role
            )
        if self.instance_type:
            self.add_option_setting(
                namespaces.LAUNCH_CONFIGURATION,
                option_names.INSTANCE_TYPE,
                self.instance_type)
        if self.single_instance:
            self.add_option_setting(
                namespaces.ENVIRONMENT,
                option_names.ENVIRONMENT_TYPE,
                'SingleInstance')
        if self.key_name:
            self.add_option_setting(
                namespaces.LAUNCH_CONFIGURATION,
                option_names.EC2_KEY_NAME,
                self.key_name)
        if self.scale:
            self.add_option_setting(
                namespaces.AUTOSCALING,
                option_names.MAX_SIZE,
                self.scale)
            self.add_option_setting(
                namespaces.AUTOSCALING,
                option_names.MIN_SIZE,
                self.scale)
        if self.elb_type:
            self.add_option_setting(
                namespaces.ENVIRONMENT,
                option_names.LOAD_BALANCER_TYPE,
                self.elb_type)

    def add_client_defaults(self):
        if self.template_name:
            # dont add client defaults if a template is being used
            return

        if not self.instance_type:
            if ec2.has_default_vpc():
                # Launch with t2 micro if not a classic account
                self.add_option_setting(
                    namespaces.LAUNCH_CONFIGURATION,
                    option_names.INSTANCE_TYPE,
                    't2.micro'
                )
        if self.platform is not None and self.platform.has_healthd_support():
            self.add_option_setting(
                namespaces.HEALTH_SYSTEM,
                option_names.SYSTEM_TYPE,
                'enhanced')

        self.add_option_setting(
            namespaces.COMMAND,
            option_names.BATCH_SIZE,
            '30')
        self.add_option_setting(
            namespaces.COMMAND,
            option_names.BATCH_SIZE_TYPE,
            'Percentage')
        if not self.tier or self.tier.name.lower() == 'webserver':
            self.add_option_setting(
                namespaces.ELB_POLICIES,
                option_names.CONNECTION_DRAINING,
                'true')
            self.add_option_setting(
                namespaces.LOAD_BALANCER,
                option_names.CROSS_ZONE,
                'true')
            if not self.single_instance:
                self.add_option_setting(
                    namespaces.ROLLING_UPDATES,
                    option_names.ROLLING_UPDATE_ENABLED,
                    'true')
                self.add_option_setting(
                    namespaces.ROLLING_UPDATES,
                    option_names.ROLLING_UPDATE_TYPE,
                    'Health')

    def compile_database_options(self):
        if not self.database:
            return

        namespace = namespaces.RDS
        self.add_option_setting(namespace, option_names.DB_PASSWORD,
                                self.database['password'])
        self.add_option_setting(namespace, option_names.DB_USER,
                                self.database['username'])
        if self.database['instance']:
            self.add_option_setting(namespace, option_names.DB_INSTANCE,
                                    self.database['instance'])
        if self.database['size']:
            self.add_option_setting(namespace, option_names.DB_STORAGE_SIZE,
                                    self.database['size'])
        if self.database['engine']:
            self.add_option_setting(namespace, option_names.DB_ENGINE,
                                    self.database['engine'])
        if self.database['version']:
            self.add_option_setting(namespace, option_names.DB_ENGINE_VERSION,
                                    self.database['version'])
        self.add_option_setting(namespace, option_names.DB_DELETION_POLICY,
                                'Snapshot')

    def compile_vpc_options(self):
        if not self.vpc:
            return

        namespace = namespaces.VPC
        self.add_option_setting(namespace, option_names.VPC_ID,
                                self.vpc['id'])
        self.add_option_setting(namespace, option_names.PUBLIC_IP,
                                self.vpc['publicip'])
        self.add_option_setting(namespace, option_names.ELB_SCHEME,
                                self.vpc['elbscheme'])
        if self.vpc['elbsubnets']:
            self.add_option_setting(namespace, option_names.ELB_SUBNETS,
                                    self.vpc['elbsubnets'])
        if self.vpc['ec2subnets']:
            self.add_option_setting(namespace, option_names.SUBNETS,
                                    self.vpc['ec2subnets'])
        if self.vpc['securitygroups']:
            self.add_option_setting(namespaces.LAUNCH_CONFIGURATION,
                                    option_names.SECURITY_GROUPS,
                                    self.vpc['securitygroups'])
        if self.vpc['dbsubnets']:
            self.add_option_setting(namespace, option_names.DB_SUBNETS,
                                    self.vpc['dbsubnets'])


class CloneEnvironmentRequest(CreateEnvironmentRequest):
    def __init__(self, app_name=None, env_name=None, original_name=None,
                 cname=None, platform=None, scale=None, tags=None, platform_arn=None):
        if not original_name:
            raise TypeError(self.__class__.__name__ + ' requires key-word argument clone_name')
        self.original_name = original_name
        super(CloneEnvironmentRequest, self).__init__(
            app_name=app_name, env_name=env_name, cname=cname,
            platform=platform, scale=scale, tags=tags, platform_arn=platform_arn
        )
        self.description = strings['env.clonedescription']. \
            replace('{env-name}', self.env_name)

    def compile_option_settings(self):
        if not self.compiled:
            # dont compile extras like vpc/database/etc.
            self.compile_common_options()
            self.compiled = True
