#!/usr/bin/env python
from setuptools import setup
setup(
  name = 'cs.binary',
  author = 'Cameron Simpson',
  author_email = 'cs@cskk.id.au',
  version = '20210316',
  url = 'https://bitbucket.org/cameron_simpson/css/commits/all',
  description =
    ('Facilities associated with binary data parsing and transcription. The '    
 'classes in this module support easy parsing of binary data structures, '    
 'returning instances with the binary data decoded into attributes and capable '    
 'of transcribing themselves in binary form (trivially via `bytes(instance)` '    
 'and also otherwise).'),
  long_description =
    ('Facilities associated with binary data parsing and transcription.\n'    
 'The classes in this module support easy parsing of binary data\n'    
 'structures,\n'    
 'returning instances with the binary data decoded into attributes\n'    
 'and capable of transcribing themselves in binary form\n'    
 '(trivially via `bytes(instance)` and also otherwise).\n'    
 '\n'    
 '*Latest release 20210316*:\n'    
 '* BSUInt: rename parse_bytes to decode_bytes, the former name conflicted '    
 'with BinaryMixin.parse_bytes and broken the semantics.\n'    
 '* Minor refactors.\n'    
 '\n'    
 'Note: this module requires Python 3.6+ because various default\n'    
 'behaviours rely on `dict`s preserving their insert order.\n'    
 '\n'    
 'See `cs.iso14496` for an ISO 14496 (eg MPEG4) parser\n'    
 'built using this module.\n'    
 '\n'    
 '**Deprecation**: the `Packet` and `PacketField` classes\n'    
 'were unnecessarily hard to use and are deprecated\n'    
 'in favour of the `Binary`* suite of classes and factories.\n'    
 'All the *`Field` classes and other subclasses\n'    
 'derived from `Packet` and `PacketField` are also deprecated.\n'    
 '\n'    
 'Terminology used below:\n'    
 '* buffer:\n'    
 '  an instance of `cs.buffer.CornuCopyBuffer`,\n'    
 '  which presents an iterable of bytes-like values\n'    
 '  via various useful methods;\n'    
 '  it also has a few factory methods to make one from a variety of sources\n'    
 '  such as bytes, iterables, binary files, `mmap`ped files,\n'    
 '  TCP data streams, etc.\n'    
 '* chunk:\n'    
 '  a piece of binary data obeying the buffer protocol,\n'    
 '  almost always a `bytes` instance or a `memoryview`,\n'    
 '  but in principle also things like `bytearray`.\n'    
 '\n'    
 'There are 5 main classes on which an implementor should base their data '    
 'structures:\n'    
 '* `BinarySingleStruct`: a factory for classes based\n'    
 '  on a `struct.struct` format string with a single value;\n'    
 '  this builds a `namedtuple` subclass\n'    
 '* `BinaryMultiStruct`: a factory for classes based\n'    
 '  on a `struct.struct` format string with multiple values;\n'    
 '  this also builds a `namedtuple` subclass\n'    
 '* `BinarySingleValue`: a base class for subclasses\n'    
 '  parsing and transcribing a single value\n'    
 '* `BinaryMultiValue`: a base class for subclasses\n'    
 '  parsing and transcribing multiple values\n'    
 '  with no variation\n'    
 '* `SimpleBinary`: a base class for subclasses\n'    
 '  with custom `.parse` and `.transcribe` methods,\n'    
 '  for structures with variable fields\n'    
 '\n'    
 'All the classes derived from the above inherit all the methods\n'    
 'of `BinaryMixin`.\n'    
 'Amongst other things, this means that the binary transcription\n'    
 'can be had simply from `bytes(instance)`,\n'    
 'although there are more transcription methods provided\n'    
 'for when greater flexibility is desired.\n'    
 'It also means that all classes have `parse`* and `scan`* methods\n'    
 'for parsing binary data streams.\n'    
 '\n'    
 'You can also instantiate objects directly;\n'    
 "there's no requirement for the source information to be binary.\n"    
 '\n'    
 'There are several presupplied subclasses for common basic types\n'    
 'such as `UInt32BE` (an unsigned 32 bit big endian integer).\n'    
 '\n'    
 '## Class `AbstractBinary(BinaryMixin)`\n'    
 '\n'    
 'Abstract class for all `Binary`* implementations,\n'    
 'specifying the `parse` and `transcribe` methods\n'    
 'and providing the methods from `BinaryMixin`.\n'    
 '\n'    
 '### Method `AbstractBinary.parse(bfr)`\n'    
 '\n'    
 'Parse an instance of `cls` from the buffer `bfr`.\n'    
 '\n'    
 '### Method `AbstractBinary.transcribe(self)`\n'    
 '\n'    
 'Return or yield `bytes`, ASCII string, `None` or iterables\n'    
 'comprising the binary form of this instance.\n'    
 '\n'    
 'This aims for maximum convenience\n'    
 'when transcribing a data structure.\n'    
 '\n'    
 'This may be implemented as a generator, yielding parts of the structure.\n'    
 '\n'    
 'This may be implemented as a normal function, returning:\n'    
 '* `None`: no bytes of data,\n'    
 '  for example for an omitted or empty structure\n'    
 '* a `bytes`-like object: the full data bytes for the structure\n'    
 '* an ASCII compatible string:\n'    
 "  this will be encoded with the `'ascii'` encoding to make `bytes`\n"    
 '* an iterable:\n'    
 '  the components of the structure,\n'    
 '  including substranscriptions which themselves\n'    
 '  adhere to this protocol - they may be `None`, `bytes`-like objects,\n'    
 '  ASCII compatible strings or iterables.\n'    
 "  This supports directly returning or yielding the result of a field's\n"    
 '  `.transcribe` method.\n'    
 '\n'    
 '## Class `BinaryByteses(AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A list of `bytes` parsed directly from the native iteration of the buffer.\n'    
 '\n'    
 '## Function `BinaryFixedBytes(class_name, length: int)`\n'    
 '\n'    
 'Factory for an `AbstractBinary` subclass matching `length` bytes of data.\n'    
 'The bytes are saved as the attribute `.data`.\n'    
 '\n'    
 '## Class `BinaryListValues(AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A list of values with a common parse specification,\n'    
 'such as sample or Boxes in an ISO14496 Box structure.\n'    
 '\n'    
 '### Method `BinaryListValues.parse(bfr, count=None, *, end_offset=None, '    
 'min_count=None, max_count=None, pt)`\n'    
 '\n'    
 'Read values from `bfr`.\n'    
 'Return a `BinaryListValue` containing the values.\n'    
 '\n'    
 'Parameters:\n'    
 '* `count`: optional count of values to read;\n'    
 '  if specified, exactly this many values are expected.\n'    
 '* `end_offset`: an optional bounding end offset of the buffer.\n'    
 '* `min_count`: the least acceptable number of values.\n'    
 '* `max_count`: the most acceptable number of values.\n'    
 '* `pt`: a parse/transcribe specification\n'    
 '  as accepted by the `pt_spec()` factory.\n'    
 '  The values will be returned by its parse function.\n'    
 '\n'    
 '### Method `BinaryListValues.transcribe(self)`\n'    
 '\n'    
 'Transcribe all the values.\n'    
 '\n'    
 '## Class `BinaryMixin`\n'    
 '\n'    
 'Presupplied helper methods for binary objects.\n'    
 '\n'    
 '### Method `BinaryMixin.__bytes__(self)`\n'    
 '\n'    
 'The binary transcription as a single `bytes` object.\n'    
 '\n'    
 '### Method `BinaryMixin.__len__(self)`\n'    
 '\n'    
 'Compute the length by running a transcription and measuring it.\n'    
 '\n'    
 '### Method `BinaryMixin.from_bytes(bs, **kw)`\n'    
 '\n'    
 'Factory to parse an instance from the\n'    
 'bytes `bs` starting at `offset`.\n'    
 'Returns the new instance.\n'    
 '\n'    
 'Raises `ValueError` if `bs` is not entirely consumed.\n'    
 'Raises `EOFError` if `bs` has insufficient data.\n'    
 '\n'    
 'Keyword parameters are passed to the `.parse_bytes` method.\n'    
 '\n'    
 'This relies on the `cls.parse` method for the parse.\n'    
 '\n'    
 '### Method `BinaryMixin.parse_bytes(bs, offset=0, length=None, **kw)`\n'    
 '\n'    
 'Factory to parse an instance from the\n'    
 'bytes `bs` starting at `offset`.\n'    
 'Returns `(instance,offset)` being the new instance and the post offset.\n'    
 '\n'    
 'Raises `EOFError` if `bs` has insufficient data.\n'    
 '\n'    
 'The parameters `offset` and `length` are passed to the\n'    
 '`CornuCopyBuffer.from_bytes` factory.\n'    
 '\n'    
 'Other keyword parameters are passed to the `.parse` method.\n'    
 '\n'    
 'This relies on the `cls.parse` method for the parse.\n'    
 '\n'    
 '### Method `BinaryMixin.scan(bfr, count=None, min_count=None, '    
 'max_count=None)`\n'    
 '\n'    
 'Function to scan the buffer `bfr` for repeated instances of `cls`\n'    
 'until end of input and yield them.\n'    
 '\n'    
 'Parameters:\n'    
 '* `bfr`: the buffer to scan\n'    
 '* `count`: the required number of instances to scan,\n'    
 '  equivalent to setting `min_count=count` and `max_count=count`\n'    
 '* `min_count`: the minimum number of instances to scan\n'    
 '* `max_count`: the maximum number of instances to scan\n'    
 'It is in error to specify both `count` and one of `min_count` or '    
 '`max_count`.\n'    
 '\n'    
 'Scanning stops after `max_count` instances (if specified).\n'    
 'If fewer than `min_count` instances (if specified) are scanned\n'    
 'a warning is issued.\n'    
 'This is to accomodate nonconformant streams\n'    
 'without raising exceptions.\n'    
 'Callers wanting to validate `max_count` may want to probe `bfr.at_eof()`\n'    
 'after return.\n'    
 'Callers not wanting a warning over `min_count` should not specify it,\n'    
 'and instead check the number of instances returned themselves.\n'    
 '\n'    
 '### Method `BinaryMixin.scan_file(f)`\n'    
 '\n'    
 'Function to scan the file `f` for repeated instances of `cls`\n'    
 'until end of input,\n'    
 'yields instances of `f`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `f`: the binary file object to parse;\n'    
 '  if `f` is a string, that pathname is opened for binary read.\n'    
 '\n'    
 '### Method `BinaryMixin.scan_with_offsets(bfr, count=None, min_count=None, '    
 'max_count=None)`\n'    
 '\n'    
 'Wrapper for `scan()` which yields (pre_offset,instance,post_offset)`\n'    
 'indicating the start and end offsets of the yielded instances.\n'    
 'All parameters are as for `scan()`.\n'    
 '\n'    
 '### Method `BinaryMixin.self_check(self, *a, **kw)`\n'    
 '\n'    
 'Internal self check. Returns `True` if passed.\n'    
 '\n'    
 'If the structure has a `FIELD_TYPES` attribute, normally a\n'    
 'class attribute, then check the fields against it. The\n'    
 '`FIELD_TYPES` attribute is a mapping of `field_name` to\n'    
 'a specification of `required` and `types`. The specification\n'    
 'may take one of 2 forms:\n'    
 '* a tuple of `(required,types)`\n'    
 '* a single `type`; this is equivalent to `(True,(type,))`\n'    
 'Their meanings are as follows:\n'    
 '* `required`: a Boolean. If true, the field must be present\n'    
 '  in the packet `field_map`, otherwise it need not be present.\n'    
 '* `types`: a tuple of acceptable field types\n'    
 '\n'    
 'There are some special semantics involved here.\n'    
 '\n'    
 'An implementation of a structure may choose to make some\n'    
 'fields plain instance attributes instead of binary objects\n'    
 'in the `field_map` mapping, particularly variable structures\n'    
 'such as a `cs.iso14496.BoxHeader`, whose `.length` may be parsed\n'    
 'directly from its binary form or computed from other fields\n'    
 'depending on the `box_size` value. Therefore, checking for\n'    
 'a field is first done via the `field_map` mapping, then by\n'    
 '`getattr`, and as such the acceptable `types` may include\n'    
 'nonstructure types such as `int`.\n'    
 '\n'    
 'Here is the `BoxHeader.FIELD_TYPES` definition as an example:\n'    
 '\n'    
 '    FIELD_TYPES = {\n'    
 "      'box_size': UInt32BE,\n"    
 "      'box_type': BytesField,\n"    
 "      'length': (\n"    
 '          True,\n'    
 '          (\n'    
 '              type(Ellipsis),\n'    
 '              UInt64BE,\n'    
 '              UInt32BE,\n'    
 '              int\n'    
 '          ),\n'    
 '      ),\n'    
 '    }\n'    
 '\n'    
 'Note that `length` includes some nonstructure types,\n'    
 'and that it is written as a tuple of `(True,types)` because\n'    
 'it has more than one acceptable type.\n'    
 '\n'    
 '### Method `BinaryMixin.transcribe_flat(self)`\n'    
 '\n'    
 'Return a flat iterable of chunks transcribing this field.\n'    
 '\n'    
 '### Method `BinaryMixin.transcribed_length(self)`\n'    
 '\n'    
 'Compute the length by running a transcription and measuring it.\n'    
 '\n'    
 '## Function `BinaryMultiStruct(class_name: str, struct_format: str, '    
 'field_names: str)`\n'    
 '\n'    
 'A class factory for `AbstractBinary` `namedtuple` subclasses\n'    
 'built around complex `struct` formats.\n'    
 '\n'    
 'Parameters:\n'    
 '* `class_name`: name for the generated class\n'    
 '* `struct_format`: the `struct` format string\n'    
 '* `field_names`: field name list,\n'    
 '  a space separated string or an interable of strings\n'    
 '\n'    
 '## Function `BinaryMultiValue(class_name, field_map, field_order=None)`\n'    
 '\n'    
 'Construct a `SimpleBinary` subclass named `class_name`\n'    
 'whose fields are specified by the mapping `field_map`.\n'    
 '\n'    
 'The `field_map` is a mapping of field name to buffer parsers and '    
 'transcribers.\n'    
 '\n'    
 '*Note*:\n'    
 'if `field_order` is not specified\n'    
 'it is constructed by iterating over `field_map`.\n'    
 'Prior to Python 3.6, `dict`s do not provide a reliable order\n'    
 'and should be accompanied by an explicit `field_order`.\n'    
 'From 3.6 onward a `dict` is enough and its insertion order\n'    
 'will dictate the default `field_order`.\n'    
 '\n'    
 'For a fixed record structure\n'    
 'the default `.parse` and `.transcribe` methods will suffice;\n'    
 'they parse or transcribe each field in turn.\n'    
 'Subclasses with variable records should override\n'    
 'the `.parse` and `.transcribe` methods\n'    
 'accordingly.\n'    
 '\n'    
 'The `field_map` is a mapping of field name\n'    
 'to a class returned by the `pt_spec()` function.\n'    
 '\n'    
 'If the class has both `parse_value` and `transcribe_value` methods\n'    
 'then the value itself will be directly stored.\n'    
 'Otherwise the class it presumed to be more complex subclass\n'    
 'of `AbstractBinary` and the instance is stored.\n'    
 '\n'    
 'Here is an example exhibiting various ways of defining each field:\n'    
 '* `n1`: defined with the *`_value` methods of `UInt8`,\n'    
 '  which return or transcribe the `int` from an unsigned 8 bit value;\n'    
 '  this stores a `BinarySingleValue` whose `.value` is an `int`\n'    
 '* `n2`: defined from the `UInt8` class,\n'    
 '  which parses an unsigned 8 bit value;\n'    
 '  this stores an `UInt8` instance\n'    
 '  (also a `BinarySingleValue` whole `.value` is an `int`)\n'    
 '* `n3`: like `n2`\n'    
 '* `data1`: defined with the *`_value` methods of `BSData`,\n'    
 '  which return or transcribe the data `bytes`\n'    
 '  from a run length encoded data chunk;\n'    
 '  this stores a `BinarySingleValue` whose `.value` is a `bytes`\n'    
 '* `data2`: defined from the `BSData` class\n'    
 '  which parses a run length encoded data chunk;\n'    
 '  this is a `BinarySingleValue` so we store its `bytes` value directly.\n'    
 '\n'    
 '    >>> class BMV(BinaryMultiValue("BMV", {\n'    
 "    ...         'n1': (UInt8.parse_value, UInt8.transcribe_value),\n"    
 "    ...         'n2': UInt8,\n"    
 "    ...         'n3': UInt8,\n"    
 "    ...         'nd': ('>H4s', 'short bs'),\n"    
 "    ...         'data1': (\n"    
 '    ...             BSData.parse_value,\n'    
 '    ...             BSData.transcribe_value,\n'    
 '    ...         ),\n'    
 "    ...         'data2': BSData,\n"    
 '    ... })):\n'    
 '    ...     pass\n'    
 '    >>> BMV.FIELD_ORDER\n'    
 "    ['n1', 'n2', 'n3', 'nd', 'data1', 'data2']\n"    
 '    >>> bmv = '    
 "BMV.from_bytes(b'\\x11\\x22\\x77\\x81\\x82zyxw\\x02AB\\x04DEFG')\n"    
 '    >>> bmv.n1  #doctest: +ELLIPSIS\n'    
 '    17\n'    
 '    >>> bmv.n2\n'    
 '    34\n'    
 '    >>> bmv  #doctest: +ELLIPSIS\n'    
 "    BMV(n1=17, n2=34, n3=119, nd=nd_1_short__bs(short=33154, bs=b'zyxw'), "    
 "data1=b'AB', data2=b'DEFG')\n"    
 '    >>> bmv.nd  #doctest: +ELLIPSIS\n'    
 "    nd_1_short__bs(short=33154, bs=b'zyxw')\n"    
 '    >>> bmv.nd.bs\n'    
 "    b'zyxw'\n"    
 '    >>> bytes(bmv.nd)\n'    
 "    b'\x81\x82zyxw'\n"    
 '    >>> bmv.data1\n'    
 "    b'AB'\n"    
 '    >>> bmv.data2\n'    
 "    b'DEFG'\n"    
 '    >>> bytes(bmv)\n'    
 '    b\'\\x11"w\\x81\\x82zyxw\\x02AB\\x04DEFG\'\n'    
 '    >>> list(bmv.transcribe_flat())\n'    
 '    [b\'\\x11\', b\'"\', b\'w\', b\'\\x81\\x82zyxw\', b\'\\x02\', b\'AB\', '    
 "b'\\x04', b'DEFG']\n"    
 '\n'    
 '## Function `BinarySingleStruct(class_name, struct_format, '    
 'field_name=None)`\n'    
 '\n'    
 'A convenience wrapper for `BinaryMultiStruct`\n'    
 'for `struct_format`s with a single field.\n'    
 '\n'    
 'Parameters:\n'    
 '* `class_name`: the class name for the generated class\n'    
 '* `struct_format`: the struct format string, specifying a\n'    
 '  single struct field\n'    
 '* `field_name`: optional field name for the value,\n'    
 "  default `'value'`\n"    
 '\n'    
 'Example:\n'    
 '\n'    
 "    >>> UInt16BE = BinarySingleStruct('UInt16BE', '>H')\n"    
 '    >>> UInt16BE.__name__\n'    
 "    'UInt16BE'\n"    
 '    >>> UInt16BE.format\n'    
 "    '>H'\n"    
 '    >>> UInt16BE.struct   #doctest: +ELLIPSIS\n'    
 '    <_struct.Struct object at ...>\n'    
 '    >>> field = UInt16BE.from_bytes(bytes((2,3)))\n'    
 '    >>> field\n'    
 '    UInt16BE(value=515)\n'    
 '    >>> field.value\n'    
 '    515\n'    
 '\n'    
 '## Class `BinarySingleValue(AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A representation of a single value as the attribute `.value`.\n'    
 '\n'    
 'Subclasses must implement:\n'    
 '* `parse` or `parse_value`\n'    
 '* `transcribe` or `transcribe_value`\n'    
 '\n'    
 '### Method `BinarySingleValue.parse(bfr)`\n'    
 '\n'    
 'Parse an instance from `bfr`.\n'    
 '\n'    
 'Subclasses must implement this method or `parse_value`.\n'    
 '\n'    
 '### Method `BinarySingleValue.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr` based on this class.\n'    
 '\n'    
 'Subclasses must implement this method or `parse`.\n'    
 '\n'    
 '### Method `BinarySingleValue.parse_value_from_bytes(bs, offset=0, '    
 'length=None, **kw)`\n'    
 '\n'    
 'Parse a value from the bytes `bs` based on this class.\n'    
 'Return `(value,offset)`.\n'    
 '\n'    
 '### Method `BinarySingleValue.scan_values(bfr, **kw)`\n'    
 '\n'    
 'Scan `bfr`, yield values.\n'    
 '\n'    
 '### Method `BinarySingleValue.transcribe(self)`\n'    
 '\n'    
 'Transcribe this instance as bytes.\n'    
 '\n'    
 'Subclasses must implement this method or `transcribe_value`.\n'    
 '\n'    
 '### Method `BinarySingleValue.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe `value` as bytes based on this class.\n'    
 '\n'    
 'Subclasses must implement this method or `transcribe`.\n'    
 '\n'    
 '## Class `BinaryUTF16NUL(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A NUL terminated UTF-16 string.\n'    
 '\n'    
 '### Method `BinaryUTF16NUL.__init__(self, value, *, encoding)`\n'    
 '\n'    
 'pylint: disable=super-init-not-called\n'    
 '\n'    
 '### Method `BinaryUTF16NUL.parse(bfr, *, encoding)`\n'    
 '\n'    
 'Parse the encoding and value and construct an instance.\n'    
 '\n'    
 '### Method `BinaryUTF16NUL.parse_value(bfr, *, encoding)`\n'    
 '\n'    
 'Read a NUL terminated UTF-16 string from `bfr`, return a `UTF16NULField`..\n'    
 'The mandatory parameter `encoding` specifies the UTF16 encoding to use\n'    
 "(`'utf_16_be'` or `'utf_16_le'`).\n"    
 '\n'    
 '### Method `BinaryUTF16NUL.transcribe(self)`\n'    
 '\n'    
 'Transcribe `self.value` in UTF-16 with a terminating NUL.\n'    
 '\n'    
 "### Method `BinaryUTF16NUL.transcribe_value(value, encoding='utf-16')`\n"    
 '\n'    
 'Transcribe `value` in UTF-16 with a terminating NUL.\n'    
 '\n'    
 '## Class `BinaryUTF8NUL(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A NUL terminated UTF-8 string.\n'    
 '\n'    
 '### Method `BinaryUTF8NUL.parse_value(bfr)`\n'    
 '\n'    
 'Read a NUL terminated UTF-8 string from `bfr`, return field.\n'    
 '\n'    
 '### Method `BinaryUTF8NUL.transcribe_value(s)`\n'    
 '\n'    
 'Transcribe the `value` in UTF-8 with a terminating NUL.\n'    
 '\n'    
 '## Class `BSData(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A run length encoded data chunk, with the length encoded as a `BSUInt`.\n'    
 '\n'    
 '### Property `BSData.data`\n'    
 '\n'    
 'An alias for the `.value` attribute.\n'    
 '\n'    
 '### Property `BSData.data_offset`\n'    
 '\n'    
 'The length of the length indicator,\n'    
 'useful for computing the location of the raw data.\n'    
 '\n'    
 '### Method `BSData.data_offset_for(bs)`\n'    
 '\n'    
 'Compute the `data_offset` which would obtain for the bytes `bs`.\n'    
 '\n'    
 '### Method `BSData.parse_value(bfr)`\n'    
 '\n'    
 'Parse the data from `bfr`.\n'    
 '\n'    
 '### Method `BSData.transcribe_value(data)`\n'    
 '\n'    
 'Transcribe the payload length and then the payload.\n'    
 '\n'    
 '## Class `BSSFloat(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A float transcribed as a BSString of str(float).\n'    
 '\n'    
 '### Method `BSSFloat.parse_value(bfr)`\n'    
 '\n'    
 'Parse a BSSFloat from a buffer and return the float.\n'    
 '\n'    
 '### Method `BSSFloat.transcribe_value(f)`\n'    
 '\n'    
 'Transcribe a float.\n'    
 '\n'    
 '## Class `BSString(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A run length encoded string, with the length encoded as a BSUInt.\n'    
 '\n'    
 "### Method `BSString.parse_value(bfr, encoding='utf-8', errors='strict')`\n"    
 '\n'    
 'Parse a run length encoded string from `bfr`.\n'    
 '\n'    
 "### Method `BSString.transcribe_value(s, encoding='utf-8')`\n"    
 '\n'    
 'Transcribe a string.\n'    
 '\n'    
 '## Class `BSUInt(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A binary serialised unsigned `int`.\n'    
 '\n'    
 'This uses a big endian byte encoding where continuation octets\n'    
 'have their high bit set. The bits contributing to the value\n'    
 'are in the low order 7 bits.\n'    
 '\n'    
 '### Method `BSUInt.decode_bytes(data, offset=0)`\n'    
 '\n'    
 'Decode an extensible byte serialised unsigned `int` from `data` at '    
 '`offset`.\n'    
 'Return value and new offset.\n'    
 '\n'    
 'Continuation octets have their high bit set.\n'    
 'The octets are big-endian.\n'    
 '\n'    
 "If you just have a `bytes` instance, this is the go. If you're\n"    
 "reading from a stream you're better off with `parse` or `parse_value`.\n"    
 '\n'    
 'Examples:\n'    
 '\n'    
 "    >>> BSUInt.decode_bytes(b'\\0')\n"    
 '    (0, 1)\n'    
 '\n'    
 'Note: there is of course the usual `BinaryMixin.parse_bytes`\n'    
 'but that constructs a buffer to obtain the individual bytes;\n'    
 'this static method will be more performant\n'    
 'if all you are doing is reading this serialisation\n'    
 'and do not already have a buffer.\n'    
 '\n'    
 '### Method `BSUInt.parse_value(bfr)`\n'    
 '\n'    
 'Parse an extensible byte serialised unsigned `int` from a buffer.\n'    
 '\n'    
 'Continuation octets have their high bit set.\n'    
 'The value is big-endian.\n'    
 '\n'    
 'This is the go for reading from a stream. If you already have\n'    
 'a bare bytes instance then the `.decode_bytes` static method\n'    
 'is probably most efficient;\n'    
 'there is of course the usual `BinaryMixin.parse_bytes`\n'    
 'but that constructs a buffer to obtain the individual bytes.\n'    
 '\n'    
 '### Method `BSUInt.transcribe_value(n)`\n'    
 '\n'    
 'Encode an unsigned int as an entensible byte serialised octet\n'    
 'sequence for decode. Return the bytes object.\n'    
 '\n'    
 '## Class `BytesesField(PacketField)`\n'    
 '\n'    
 'A field containing a list of bytes chunks.\n'    
 '\n'    
 'The following attributes are defined:\n'    
 '* `value`: the gathered data as a list of bytes instances,\n'    
 '  or None if the field was gathered with `discard_data` true.\n'    
 '* `offset`: the starting offset of the data.\n'    
 '* `end_offset`: the ending offset of the data.\n'    
 '\n'    
 'The `offset` and `end_offset` values are recorded during the\n'    
 "parse, and may become irrelevant if the field's contents are\n"    
 'changed.\n'    
 '\n'    
 '### Method `BytesesField.from_buffer(bfr, end_offset=None, '    
 'discard_data=False, short_ok=False)`\n'    
 '\n'    
 'Create a new `BytesesField` from a buffer\n'    
 'by gathering from `bfr` until `end_offset`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `bfr`: the input buffer\n'    
 '* `end_offset`: the ending buffer offset; if this is Ellipsis\n'    
 '  then all the remaining data in `bfr` will be collected\n'    
 '* `discard_data`: discard the data, keeping only the offset information\n'    
 '* `short_ok`: if true, do not raise EOFError if there are\n'    
 "  insufficient data; the field's .end_offset value will be\n"    
 '  less than `end_offset`; the default is False\n'    
 '\n'    
 'Note that this method also sets the following attributes\n'    
 'on the new `BytesesField`:\n'    
 '* `offset`: the starting offset of the gathered bytes\n'    
 '* `end_offset`: the ending offset after the gathered bytes\n'    
 '* `length`: the length of the data\n'    
 '\n'    
 '## Class `BytesField(BinarySingleValue,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'A field of bytes.\n'    
 '\n'    
 '### Method `BytesField.__len__(self)`\n'    
 '\n'    
 'The length is the length of the data.\n'    
 '\n'    
 '### Property `BytesField.data`\n'    
 '\n'    
 'Alias for the `.value` attribute.\n'    
 '\n'    
 '### Property `BytesField.length`\n'    
 '\n'    
 'Convenient length attribute.\n'    
 '\n'    
 '### Method `BytesField.transcribe_value(value)`\n'    
 '\n'    
 'A `BytesField` is its own transcription.\n'    
 '\n'    
 '### Method `BytesField.value_from_buffer(bfr, length=None)`\n'    
 '\n'    
 'Parse a `BytesField` of length `length` from `bfr`.\n'    
 '\n'    
 '## Class `BytesRunField(PacketField)`\n'    
 '\n'    
 'A field containing a continuous run of a single bytes value.\n'    
 '\n'    
 'The following attributes are defined:\n'    
 '* `length`: the length of the run\n'    
 '* `bytes_value`: the repeated bytes value\n'    
 '\n'    
 'The property `value` is computed on the fly on every reference\n'    
 'and returns a value obeying the buffer protocol: a bytes or\n'    
 'memoryview object.\n'    
 '\n'    
 '### Method `BytesRunField.__init__(self, length, bytes_value)`\n'    
 '\n'    
 'pylint: disable=super-init-not-called\n'    
 '\n'    
 '### Method `BytesRunField.from_buffer(bfr, end_offset=None, '    
 "bytes_value=b'\\x00')`\n"    
 '\n'    
 'Parse a BytesRunField by just skipping the specified number of bytes.\n'    
 '\n'    
 'Note: this *does not* check that the skipped bytes contain `bytes_value`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `bfr`: the buffer to scan\n'    
 '* `end_offset`: the end offset of the scan, which may be\n'    
 '  an int or Ellipsis to indicate a scan to the end of the\n'    
 '  buffer\n'    
 '* `bytes_value`: the bytes value to replicate, default\n'    
 "  `b'\x00'`; if this is an int then a single byte of that value\n"    
 '  is used\n'    
 '\n'    
 '### Method `BytesRunField.transcribe(self)`\n'    
 '\n'    
 'Transcribe the BytesRunField in 256 byte chunks.\n'    
 '\n'    
 '### Property `BytesRunField.value`\n'    
 '\n'    
 'The run of bytes, computed on the fly.\n'    
 '\n'    
 'Values where length <= 256 are cached.\n'    
 '\n'    
 '## Function `deferred_field(from_buffer)`\n'    
 '\n'    
 'A decorator for a field property.\n'    
 '\n'    
 'Usage:\n'    
 '\n'    
 '    @deferred_field\n'    
 '    def (self, bfr):\n'    
 '        ... parse value from `bfr`, return value\n'    
 '\n'    
 '## `EmptyField = <cs.binary.EmptyPacketField object at 0x10538c160>`\n'    
 '\n'    
 'An empty data field, used as a placeholder for optional\n'    
 'fields when they are not present.\n'    
 '\n'    
 'The singleton `EmptyField` is a predefined instance.\n'    
 '\n'    
 '## Class `EmptyPacketField(PacketField)`\n'    
 '\n'    
 'An empty data field, used as a placeholder for optional\n'    
 'fields when they are not present.\n'    
 '\n'    
 'The singleton `EmptyField` is a predefined instance.\n'    
 '\n'    
 '### Method `EmptyPacketField.from_buffer(bfr)`\n'    
 '\n'    
 'pylint: disable=arguments-differ\n'    
 '\n'    
 '## Function `fixed_bytes_field(length, class_name=None)`\n'    
 '\n'    
 'Factory for `BytesField` subclasses built from fixed length byte strings.\n'    
 '\n'    
 '## Function `flatten(chunks)`\n'    
 '\n'    
 'Flatten `chunks` into an iterable of `bytes` instances.\n'    
 '\n'    
 'This exists to allow subclass methods to easily return\n'    
 'transcribeable things (having a `.transcribe` method), ASCII\n'    
 'strings or bytes or iterables or even `None`, in turn allowing\n'    
 "them simply to return their superclass' chunks iterators\n"    
 'directly instead of having to unpack them.\n'    
 '\n'    
 'An example from the `cs.iso14496.METABoxBody` class:\n'    
 '\n'    
 '    def transcribe(self):\n'    
 '        yield super().transcribe()\n'    
 '        yield self.theHandler\n'    
 '        yield self.boxes\n'    
 '\n'    
 'The binary classes `flatten` the result of the `.transcribe`\n'    
 "method to obtain `bytes` insteances for the object's bnary\n"    
 'transcription.\n'    
 '\n'    
 '## Class `Float64BE(Float64BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>d'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Float64BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Float64BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Float64BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Float64BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `Float64LE(Float64LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<d'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Float64LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Float64LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Float64LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Float64LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `Int16BE(Int16BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>h'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Int16BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Int16BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Int16BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Int16BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `Int16LE(Int16LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<h'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Int16LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Int16LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Int16LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Int16LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `Int32BE(Int32BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>l'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Int32BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Int32BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Int32BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Int32BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `Int32LE(Int32LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<l'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `Int32LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `Int32LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `Int32LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `Int32LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `ListField(PacketField)`\n'    
 '\n'    
 'A field which is itself a list of other `PacketField`s.\n'    
 '\n'    
 '### Method `ListField.__iter__(self)`\n'    
 '\n'    
 'Iterating over a `ListField` iterates over its `.value`.\n'    
 '\n'    
 '### Method `ListField.from_buffer(bfr)`\n'    
 '\n'    
 'ListFields do not know enough to parse a buffer.\n'    
 '\n'    
 '### Method `ListField.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe each item in `value`.\n'    
 '\n'    
 '## Function `multi_struct_field(struct_format, subvalue_names=None, '    
 'class_name=None)`\n'    
 '\n'    
 'A class factory for `PacketField` subclasses built around complex `struct` '    
 'formats.\n'    
 '\n'    
 '**Deprecated**: see the `BinaryMultiValue` factory instead.\n'    
 '\n'    
 'See also the convenience class factory `structtuple`\n'    
 'which is usually easier to work with.\n'    
 '\n'    
 'Parameters:\n'    
 '* `struct_format`: the `struct` format string\n'    
 '* `subvalue_names`: an optional field name list;\n'    
 '  if supplied then the field value will be a `namedtuple` with\n'    
 '  these names\n'    
 '* `class_name`: option name for the generated class\n'    
 '\n'    
 '## Class `Packet(PacketField)`\n'    
 '\n'    
 'Base class for compound objects derived from binary data.\n'    
 '\n'    
 '*DEPRECATED*:\n'    
 'please adopt one of the `BinaryMutli`* classes instead.\n'    
 '\n'    
 '### Method `Packet.__init__(self, **fields)`\n'    
 '\n'    
 'Initialise the `Packet`.\n'    
 '\n'    
 'A `Packet` is its own `.value`.\n'    
 '\n'    
 'If any keyword arguments are provided, they are used as a\n'    
 'mapping of `field_name` to `Field` instance, supporting\n'    
 'direct construction of simple `Packet`s.\n'    
 'From Python 3.6 onwards keyword arguments preserve the calling order;\n'    
 'in Python versions earlier than this the caller should\n'    
 'adjust the `Packet.field_names` list to the correct order after\n'    
 'initialisation.\n'    
 '\n'    
 '### Method `Packet.__getattr__(self, attr)`\n'    
 '\n'    
 'Unknown attributes may be field names; return their value.\n'    
 '\n'    
 '### Method `Packet.add_deferred_field(self, attr_name, bfr, length)`\n'    
 '\n'    
 'Store the unparsed data for attribute `attr_name`\n'    
 'comprising the next `length` bytes from `bfr`.\n'    
 '\n'    
 '### Method `Packet.add_field(self, field_name, field)`\n'    
 '\n'    
 'Add a new `PacketField` `field` named `field_name`.\n'    
 '\n'    
 '### Method `Packet.add_from_buffer(self, field_name, bfr, factory, **kw)`\n'    
 '\n'    
 'Add a new `PacketField` named `field_name` parsed from `bfr` using '    
 '`factory`.\n'    
 'Updates the internal field records.\n'    
 "Returns the new `PacketField`'s .value.\n"    
 '\n'    
 'Parameters:\n'    
 '* `field_name`: the name for the new field; it must be new.\n'    
 '* `bfr`: a `CornuCopyBuffer` from which to parse the field data.\n'    
 '* `factory`: a factory for parsing the field data, returning\n'    
 '  a `PacketField`. If `factory` is a class then its .from_buffer\n'    
 '  method is called, otherwise the factory is called directly.\n'    
 '\n'    
 'Additional keyword arguments are passed to the internal\n'    
 'factory call.\n'    
 '\n'    
 'For convenience, `factory` may also be a str in which case\n'    
 'it is taken to be a single struct format specifier.\n'    
 'Alternatively, `factory` may be an integer in which case\n'    
 'it is taken to be a fixed length bytes field.\n'    
 '\n'    
 '### Method `Packet.add_from_bytes(self, field_name, bs, factory, offset=0, '    
 'length=None, **kw)`\n'    
 '\n'    
 'Add a new `PacketField` named `field_name` parsed from the\n'    
 'bytes `bs` using `factory`. Updates the internal field\n'    
 'records.\n'    
 "Returns the new `PacketField`'s .value and the new parse\n"    
 'offset within `bs`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `field_name`: the name for the new field; it must be new.\n'    
 '* `bs`: the bytes containing the field data; a `CornuCopyBuffer`\n'    
 '  is made from this for the parse.\n'    
 '* `factory`: a factory for parsing the field data, returning\n'    
 '  a `PacketField`. If `factory` is a class then its .from_buffer\n'    
 '  method is called, otherwise the factory is called directly.\n'    
 '* `offset`: optional start offset of the field data within\n'    
 '  `bs`, default 0.\n'    
 '* `length`: optional maximum number of bytes from `bs` to\n'    
 '  make available for the parse, default None meaning that\n'    
 '  everything from `offset` onwards is available.\n'    
 '\n'    
 'Additional keyword arguments are passed to the internal\n'    
 '`.add_from_buffer` call.\n'    
 '\n'    
 '### Method `Packet.add_from_value(self, field_name, value, '    
 'transcribe_value_fn)`\n'    
 '\n'    
 'Add a new field named `field_name` with `.value=value`.\n'    
 'Return the new field.\n'    
 '\n'    
 '### Method `Packet.deferred_field(from_buffer)`\n'    
 '\n'    
 'A decorator for a field property.\n'    
 '\n'    
 'Usage:\n'    
 '\n'    
 '    @deferred_field\n'    
 '    def (self, bfr):\n'    
 '        ... parse value from `bfr`, return value\n'    
 '\n'    
 '### Method `Packet.get_field(self, field_name)`\n'    
 '\n'    
 'Return the field named `field_name`.\n'    
 '\n'    
 '### Method `Packet.pop_field(self)`\n'    
 '\n'    
 'Remove the last field, return `(field_name,field)`.\n'    
 '\n'    
 '### Method `Packet.remove_field(self, field_name)`\n'    
 '\n'    
 'Remove the field `field_name`. Return the field.\n'    
 '\n'    
 '### Method `Packet.self_check(self)`\n'    
 '\n'    
 'Internal self check. Returns `True` if passed.\n'    
 '\n'    
 'If the `Packet` has a `PACKET_FIELDS` attribute, normally a\n'    
 'class attribute, then check the fields against it. The\n'    
 '`PACKET_FIELDS` attribute is a mapping of `field_name` to\n'    
 'a specification of `required` and `types`. The specification\n'    
 'may take one of 2 forms:\n'    
 '* a tuple of `(required, types)`\n'    
 '* a single `type`; this is equivalent to `(True, (type,))`\n'    
 'Their meanings are as follows:\n'    
 '* `required`: a Boolean. If true, the field must be present\n'    
 '  in the packet `field_map`, otherwise it need not be present.\n'    
 '* `types`: a tuple of acceptable field types\n'    
 '\n'    
 'There are some special semantics involved here.\n'    
 '\n'    
 'An implementation of a `Packet` may choose to make some\n'    
 'fields plain instance attributes instead of `Field`s in the\n'    
 '`field_map` mapping, particularly variable packets such as\n'    
 'a `cs.iso14496.BoxHeader`, whose `.length` may be parsed\n'    
 'directly from its binary form or computed from other fields\n'    
 'depending on the `box_size` value. Therefore, checking for\n'    
 'a field is first done via the `field_map` mapping, then by\n'    
 '`getattr`, and as such the acceptable `types` may include\n'    
 'non-`PacketField` types such as `int`.\n'    
 '\n'    
 'Here is the `BoxHeader.PACKET_FIELDS` definition as an example:\n'    
 '\n'    
 '  PACKET_FIELDS = {\n'    
 "    'box_size': UInt32BE,\n"    
 "    'box_type': BytesField,\n"    
 "    'length': (\n"    
 '        True,\n'    
 '        (\n'    
 '            type(Ellipsis),\n'    
 '            UInt64BE,\n'    
 '            UInt32BE,\n'    
 '            int\n'    
 '        ),\n'    
 '    ),\n'    
 '  }\n'    
 '\n'    
 'Note that `length` includes some non-`PacketField` types,\n'    
 'and that it is written as a tuple of `(True, types)` because\n'    
 'it has more than one acceptable type.\n'    
 '\n'    
 '### Method `Packet.set_field(self, field_name, new_field)`\n'    
 '\n'    
 'Replace the field named `field_name`.\n'    
 '\n'    
 'Note that this replaces the field, not its value.\n'    
 '\n'    
 '### Method `Packet.transcribe(self)`\n'    
 '\n'    
 'Yield a sequence of bytes objects for this instance.\n'    
 '\n'    
 '## Class `PacketField`\n'    
 '\n'    
 'A record for an individual packet field.\n'    
 '\n'    
 '*DEPRECATED*:\n'    
 'please adopt one of the `BinarySingle`* classes instead.\n'    
 '\n'    
 'This normally holds a single value,\n'    
 'for example an int of a particular size or a string.\n'    
 '\n'    
 'There are 2 basic ways to implement a `PacketField` subclass:\n'    
 '* simple: implement `value_from_buffer` and `transcribe_value`\n'    
 '* complex: implement `from_buffer` and `transcribe`\n'    
 '\n'    
 'In the simple case subclasses should implement two methods:\n'    
 '* `value_from_buffer`:\n'    
 '  parse the value from a `CornuCopyBuffer` and return the parsed value\n'    
 '* `transcribe_value`:\n'    
 '  transcribe the value as bytes\n'    
 '\n'    
 'In the more complex case,\n'    
 'sometimes a `PacketField` may not warrant (or perhaps fit)\n'    
 'the formality of a `Packet` with its multifield structure.\n'    
 '\n'    
 'One example is the `cs.iso14496.UTF8or16Field` class.\n'    
 '\n'    
 '`UTF8or16Field` supports an ISO14496 UTF8 or UTF16 string field,\n'    
 'as as such has 2 attributes:\n'    
 '* `value`: the string itself\n'    
 '* `bom`: a UTF16 byte order marker or `None`;\n'    
 '  `None` indicates that the string should be encoded as UTF-8\n'    
 '  and otherwise the BOM indicates UTF16 big endian or little endian.\n'    
 '\n'    
 'To make this subclass it defines these methods:\n'    
 '* `from_buffer`:\n'    
 '  to read the optional BOM and then the following encoded string;\n'    
 '  it then returns the new `UTF8or16Field`\n'    
 '  initialised from these values via `cls(text, bom=bom)`.\n'    
 '* `transcribe`:\n'    
 '  to transcribe the optional BOM and suitably encoded string.\n'    
 'The instance method `transcribe` is required because the transcription\n'    
 'requires knowledge of the BOM, an attribute of an instance.\n'    
 '\n'    
 '### Method `PacketField.__init__(self, value=None)`\n'    
 '\n'    
 'Initialise the `PacketField`.\n'    
 'If omitted the inial field `value` will be `None`.\n'    
 '\n'    
 '### Method `PacketField.__len__(self)`\n'    
 '\n'    
 'Compute the length by running a transcription and measuring it.\n'    
 '\n'    
 '### Method `PacketField.from_buffer(bfr, **kw)`\n'    
 '\n'    
 'Factory to return a `PacketField` instance from a `CornuCopyBuffer`.\n'    
 '\n'    
 'This default implementation is for single value `PacketField`s\n'    
 'and instantiates the instance from the value returned\n'    
 'by `cls.value_from_buffer(bfr, **kw)`;\n'    
 'implementors should implement `value_from_buffer`.\n'    
 '\n'    
 '### Method `PacketField.from_bytes(bs, offset=0, length=None, **kw)`\n'    
 '\n'    
 'Factory to return a `PacketField` instance parsed from the\n'    
 'bytes `bs` starting at `offset`.\n'    
 'Returns the new `PacketField` and the post parse offset.\n'    
 '\n'    
 'The parameters `offset` and `length` are as for the\n'    
 '`CornuCopyBuffer.from_bytes` factory.\n'    
 '\n'    
 'This relies on the `cls.from_buffer` method for the parse.\n'    
 '\n'    
 '### Method `PacketField.parse_buffer(bfr, **kw)`\n'    
 '\n'    
 'Function to parse repeated instances of `cls` from the buffer `bfr`\n'    
 'until end of input.\n'    
 '\n'    
 '### Method `PacketField.parse_buffer_values(bfr, **kw)`\n'    
 '\n'    
 'Function to parse repeated instances of `cls.value`\n'    
 'from the buffer `bfr` until end of input.\n'    
 '\n'    
 '### Method `PacketField.parse_buffer_with_offsets(bfr, **kw)`\n'    
 '\n'    
 'Function to parse repeated instances of `cls` from the buffer `bfr`\n'    
 'until end of input.\n'    
 'Yields `(offset,instance,post_offset)`\n'    
 'where `offset` if the buffer offset where the instance commenced\n'    
 'and `post_offset` is the buffer offset after the instance.\n'    
 '\n'    
 '### Method `PacketField.parse_file(f, **kw)`\n'    
 '\n'    
 'Function to parse repeated instances of `cls` from the file `f`\n'    
 'until end of input.\n'    
 '\n'    
 'Parameters:\n'    
 '* `f`: the binary file object to parse;\n'    
 '  if `f` is a string, that pathname is opened for binary read.\n'    
 '\n'    
 '### Method `PacketField.transcribe(self)`\n'    
 '\n'    
 'Return or yield the bytes transcription of this field.\n'    
 '\n'    
 'This may directly return:\n'    
 '* a `bytes` or `memoryview` holding the binary data\n'    
 '* `None`: indicating no binary data\n'    
 '* `str`: indicating the ASCII encoding of the string\n'    
 '* an iterable of these things (including further iterables)\n'    
 "  to support trivially transcribing via other fields'\n"    
 '  `transcribe` methods\n'    
 '\n'    
 'Callers will usually call `flatten` on the output of this\n'    
 'method, or use the convenience `transcribe_flat` method\n'    
 'which calls `flatten` for them.\n'    
 '\n'    
 'This default implementation is for single value fields and\n'    
 'just calls `self.transcribe_value(self.value)`.\n'    
 '\n'    
 '### Method `PacketField.transcribe_flat(self)`\n'    
 '\n'    
 'Return a flat iterable of chunks transcribing this field.\n'    
 '\n'    
 '### Method `PacketField.transcribe_value(value, **kw)`\n'    
 '\n'    
 'For simple `PacketField`s, return a bytes transcription of a\n'    
 'value suitable for the `.value` attribute.\n'    
 '\n'    
 'For example, the `BSUInt` subclass stores a `int` as its\n'    
 '`.value` and exposes its serialisation method, suitable for\n'    
 'any `int`, as `transcribe_value`.\n'    
 '\n'    
 'Note that this calls the class `transcribe` method, which\n'    
 'may return an iterable.\n'    
 'Use the `value_as_bytes` method to get a single flat `bytes` result.\n'    
 '\n'    
 '### Method `PacketField.transcribe_value_flat(value)`\n'    
 '\n'    
 'Return a flat iterable of chunks transcribing `value`.\n'    
 '\n'    
 '### Method `PacketField.value_as_bytes(value, **kw)`\n'    
 '\n'    
 'For simple `PacketField`s, return a transcription of a\n'    
 'value suitable for the `.value` attribute\n'    
 'as a single `bytes` value.\n'    
 '\n'    
 'This flattens and joins the transcription returned by\n'    
 '`transcribe_value`.\n'    
 '\n'    
 '### Method `PacketField.value_from_buffer(bfr, **kw)`\n'    
 '\n'    
 'Function to parse and return the core value from a `CornuCopyBuffer`.\n'    
 '\n'    
 'For single value fields it is enough to implement this method.\n'    
 '\n'    
 'For multiple value fields it is better to implement `cls.from_buffer`.\n'    
 '\n'    
 '### Method `PacketField.value_from_bytes(bs, offset=0, length=None, **kw)`\n'    
 '\n'    
 'Return a value parsed from the bytes `bs` starting at `offset`.\n'    
 'Returns the new value and the post parse offset.\n'    
 '\n'    
 'The parameters `offset` and `length` are as for the\n'    
 '`CornuCopyBuffer.from_bytes` factory.\n'    
 '\n'    
 'This relies on the `cls.from_bytes` method for the parse.\n'    
 '\n'    
 '### Property `PacketField.value_s`\n'    
 '\n'    
 'The preferred string representation of the value.\n'    
 '\n'    
 '## Function `pt_spec(pt, name=None)`\n'    
 '\n'    
 'Convert a parse/transcribe specification `pt`\n'    
 'into an `AbstractBinary` subclass.\n'    
 '\n'    
 'This is largely used to provide flexibility\n'    
 'in the specifications for the `BinaryMultiValue` factory\n'    
 'but can be used as a factory for other simple classes.\n'    
 '\n'    
 'If the specification `pt` is a subclass of `AbstractBinary`\n'    
 'this is returned directly.\n'    
 '\n'    
 'If `pt` is a 2-tuple of `str`\n'    
 'the values are presumed to be a format string for `struct.struct`\n'    
 'and filed names separated by spaces;\n'    
 'a new `BinaryMultiStruct` class is created from these and returned.\n'    
 '\n'    
 'Otherwise two functions\n'    
 '`f_parse_value(bfr)` and `f_transcribe_value(value)`\n'    
 'are obtained and used to construct a new `BinarySingleValue` class\n'    
 'as follows:\n'    
 '\n'    
 'If `pt` has `.parse_value` and `.transcribe_value` callable attributes,\n'    
 'use those for `f_parse_value` and `f_transcribe_value` respectively.\n'    
 '\n'    
 'Otherwise, if `pt` is an `int`\n'    
 'define `f_parse_value` to obtain exactly that many bytes from a buffer\n'    
 'and `f_transcribe_value` to return those bytes directly.\n'    
 '\n'    
 'Otherwise presume `pt` is a 2-tuple of '    
 '`(f_parse_value,f_transcribe_value)`.\n'    
 '\n'    
 '## Class `SimpleBinary(types.SimpleNamespace,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'Abstract binary class based on a `SimpleNamespace`,\n'    
 'thus providing a nice `__str__` and a keyword based `__init__`.\n'    
 'Implementors must still define `.parse` and `.transcribe`.\n'    
 '\n'    
 'To constraint the arguments passed to `__init__`,\n'    
 'define an `__init__` which accepts specific keyword arguments\n'    
 'and pass through to `super().__init__()`. Example:\n'    
 '\n'    
 '    def __init__(self, *, field1=None, field2):\n'    
 '        """ Accept only `field1` (optional)\n'    
 '            and `field2` (mandatory).\n'    
 '        """\n'    
 '        super().__init__(field1=field1, field2=field2)\n'    
 '\n'    
 '## Function `structtuple(class_name, struct_format, subvalue_names)`\n'    
 '\n'    
 'Convenience wrapper for `multi_struct_field`.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 "    Enigma2Cut = structtuple('Enigma2Cut', '>QL', 'pts type')\n"    
 '\n'    
 'which is a record with big-endian unsigned 64 and 32 fields\n'    
 'named `pts` and `type`.\n'    
 '\n'    
 '## Class `UInt16BE(UInt16BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>H'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt16BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt16BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt16BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt16BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt16LE(UInt16LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<H'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt16LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt16LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt16LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt16LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt32BE(UInt32BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>L'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt32BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt32BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt32BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt32BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt32LE(UInt32LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<L'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt32LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt32LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt32LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt32LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt64BE(UInt64BE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'>Q'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt64BE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt64BE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt64BE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt64BE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt64LE(UInt64LE,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'<Q'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt64LE.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt64LE.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt64LE.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt64LE.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UInt8(UInt8,builtins.tuple,AbstractBinary,BinaryMixin)`\n'    
 '\n'    
 'An `AbstractBinary` `namedtuple` which parses and transcribes\n'    
 "the struct format `'B'` and presents the attributes ('value',).\n"    
 '\n'    
 '### Method `UInt8.parse(bfr)`\n'    
 '\n'    
 'Parse from `bfr` via `struct.unpack`.\n'    
 '\n'    
 '### Method `UInt8.parse_value(bfr)`\n'    
 '\n'    
 'Parse a value from `bfr`, return the value.\n'    
 '\n'    
 '### Method `UInt8.transcribe(self)`\n'    
 '\n'    
 'Transcribe via `struct.pack`.\n'    
 '\n'    
 '### Method `UInt8.transcribe_value(value)`\n'    
 '\n'    
 'Transcribe a value back into bytes.\n'    
 '\n'    
 '## Class `UTF16NULField(PacketField)`\n'    
 '\n'    
 'A NUL terminated UTF-16 string.\n'    
 '\n'    
 '### Method `UTF16NULField.__init__(self, value, *, encoding)`\n'    
 '\n'    
 'Initialise the `PacketField`.\n'    
 'If omitted the inial field `value` will be `None`.\n'    
 '\n'    
 '### Method `UTF16NULField.from_buffer(bfr, encoding)`\n'    
 '\n'    
 'Read a NUL terminated UTF-16 string from `bfr`, return a `UTF16NULField`..\n'    
 'The mandatory parameter `encoding` specifies the UTF16 encoding to use\n'    
 "(`'utf_16_be'` or `'utf_16_le'`).\n"    
 '\n'    
 "### Method `UTF16NULField.transcribe_value(value, encoding='utf-16')`\n"    
 '\n'    
 'Transcribe `value` in UTF-16 with a terminating NUL.\n'    
 '\n'    
 '## Class `UTF8NULField(PacketField)`\n'    
 '\n'    
 'A NUL terminated UTF-8 string.\n'    
 '\n'    
 '### Method `UTF8NULField.transcribe_value(s)`\n'    
 '\n'    
 'Transcribe the `value` in UTF-8 with a terminating NUL.\n'    
 '\n'    
 '### Method `UTF8NULField.value_from_buffer(bfr)`\n'    
 '\n'    
 'Read a NUL terminated UTF-8 string from `bfr`, return field.\n'    
 '\n'    
 '# Release Log\n'    
 '\n'    
 '\n'    
 '\n'    
 '*Release 20210316*:\n'    
 '* BSUInt: rename parse_bytes to decode_bytes, the former name conflicted '    
 'with BinaryMixin.parse_bytes and broken the semantics.\n'    
 '* Minor refactors.\n'    
 '\n'    
 '*Release 20210306*:\n'    
 'MAJOR RELEASE: The PacketField classes and friends were hard to use; this '    
 'release supplied a suite of easier to use and more consistent Binary* '    
 'classes, and ports most of those things based on the old scheme to the new '    
 'scheme.\n'    
 '\n'    
 '*Release 20200229*:\n'    
 '* ListField: replace transcribe method with transcribe_value method, aids '    
 'external use.\n'    
 '* Add `.length` attribute to struct based packet classes providing the data '    
 'length of the structure (struct.Struct.size).\n'    
 '* Packet: new `add_deferred_field` method to consume the raw data for a '    
 'field for parsing later (done automatically if the attribute is accessed).\n'    
 '* New `@deferred_field` decorator for the parser for that stashed data.\n'    
 '\n'    
 '*Release 20191230.3*:\n'    
 'Docstring tweak.\n'    
 '\n'    
 '*Release 20191230.2*:\n'    
 'Documentation updates.\n'    
 '\n'    
 '*Release 20191230.1*:\n'    
 'Docstring updates. Semantic changes were in the previous release.\n'    
 '\n'    
 '*Release 20191230*:\n'    
 '* ListField: new __iter__ method.\n'    
 '* Packet: __str__: accept optional `skip_fields` parameter to omit some '    
 'field names.\n'    
 '* Packet: new .add_from_value method to add a named field with a presupplied '    
 'value.\n'    
 '* Packet: new remove_field(field_name) and pop_field() methods to remove '    
 'fields.\n'    
 '* BytesesField: __iter__ yields the bytes values, transcribe=__iter__.\n'    
 '* PacketField: propagate keyword arguments through various methods, required '    
 'for parameterised PacketFields.\n'    
 '* New UTF16NULField, a NUL terminated UTF16 string.\n'    
 '* PacketField: provide a default `.transcribe_value` method which makes a '    
 'new instance and calls its `.transcribe` method.\n'    
 '* Documentation update and several minor changes.\n'    
 '\n'    
 '*Release 20190220*:\n'    
 '* Packet.self_check: fields without a sanity check cause a warning, not a '    
 'ValueError.\n'    
 '* New Float64BE, Float64LE and BSSFloat classes for IEEE floats and '    
 'floats-as-strings.\n'    
 '* Additional module docstringage on subclassing Packet and PacketField.\n'    
 '* BSString: drop redundant from_buffer class method.\n'    
 '* PacketField.__init__: default to value=None if omitted.\n'    
 '\n'    
 '*Release 20181231*:\n'    
 'flatten: do not yield zero length bytelike objects, can be misread as EOF on '    
 'some streams.\n'    
 '\n'    
 '*Release 20181108*:\n'    
 '* New PacketField.transcribe_value_flat convenience method to return a flat '    
 'iterable of bytes-like objects.\n'    
 '* New PacketField.parse_buffer generator method to parse instances of the '    
 'PacketField from a buffer until end of input.\n'    
 '* New PacketField.parse_buffer_values generator method to parse instances of '    
 'the PacketField from a buffer and yield the `.value` attribute until end of '    
 'input.\n'    
 '\n'    
 '*Release 20180823*:\n'    
 '* Some bugfixes.\n'    
 '* Define PacketField.__eq__.\n'    
 '* BSUInt, BSData and BSString classes implementing the serialisations from '    
 'cs.serialise.\n'    
 '* New PacketField.value_from_bytes class method.\n'    
 '* New PacketField.value_from_buffer method.\n'    
 '\n'    
 '*Release 20180810.2*:\n'    
 'Documentation improvements.\n'    
 '\n'    
 '*Release 20180810.1*:\n'    
 'Improve module description.\n'    
 '\n'    
 '*Release 20180810*:\n'    
 "BytesesField.from_buffer: make use of the buffer's skipto method if "    
 'discard_data is true.\n'    
 '\n'    
 '*Release 20180805*:\n'    
 '* Packet: now an abstract class, new self_check method initially checking '    
 'the\n'    
 '* PACKET_FIELDS class attribute against the instance, new methods get_field\n'    
 '* and set_field to fetch or replace existing fields, allow keyword '    
 'arguments\n'    
 '* to initialise the Packet fields and document the dependency on keyword\n'    
 '* argument ordering.\n'    
 '* PacketField: __len__ computed directory from a transcribe, drop other '    
 '__len__\n'    
 '* methods.\n'    
 '* EmptyField singleton to use as a placeholder for missing optional fields.\n'    
 '* BytesField: implement value_s and from_buffer.\n'    
 '* multi_struct_field: implement __len__ for generated class.\n'    
 '* flatten: treat memoryviews like bytes.\n'    
 '* Assorted docstrings and fixes.\n'    
 '\n'    
 '*Release 20180801*:\n'    
 'Initial PyPI release.'),
  classifiers = ['Development Status :: 4 - Beta', 'Environment :: Console', 'Programming Language :: Python :: 3', 'Intended Audience :: Developers', 'Operating System :: OS Independent', 'Topic :: Software Development :: Libraries :: Python Modules', 'License :: OSI Approved :: GNU General Public License v3 or later (GPLv3+)'],
  install_requires = ['cs.buffer', 'cs.gimmicks', 'cs.lex', 'cs.pfx', 'cs.seq'],
  keywords = ['python3'],
  license = 'GNU General Public License v3 or later (GPLv3+)',
  long_description_content_type = 'text/markdown',
  package_dir = {'': 'lib/python'},
  py_modules = ['cs.binary'],
  python_requires = '>=3.6',
)
